"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addNewFeatureToIndex = addNewFeatureToIndex;
exports.clearGoto = clearGoto;
exports.clearMouseCoordinates = clearMouseCoordinates;
exports.deleteCustomIcon = deleteCustomIcon;
exports.deleteFeatureFromIndex = deleteFeatureFromIndex;
exports.mapDestroyed = mapDestroyed;
exports.mapExtentChanged = mapExtentChanged;
exports.mapReady = mapReady;
exports.rollbackMapSettings = rollbackMapSettings;
exports.setEditLayerToSelectedLayer = setEditLayerToSelectedLayer;
exports.setEmbeddableSearchContext = setEmbeddableSearchContext;
exports.setExecutionContext = setExecutionContext;
exports.setGotoWithCenter = setGotoWithCenter;
exports.setMapInitError = setMapInitError;
exports.setMapSettings = setMapSettings;
exports.setMouseCoordinates = setMouseCoordinates;
exports.setQuery = setQuery;
exports.trackMapSettings = trackMapSettings;
exports.updateCustomIcons = updateCustomIcons;
exports.updateDrawState = updateDrawState;
exports.updateEditLayer = updateEditLayer;
exports.updateEditShape = updateEditShape;
exports.updateMapSetting = updateMapSetting;
var _lodash = _interopRequireDefault(require("lodash"));
var _i18n = require("@kbn/i18n");
var _bboxPolygon = _interopRequireDefault(require("@turf/bbox-polygon"));
var _booleanContains = _interopRequireDefault(require("@turf/boolean-contains"));
var _std = require("@kbn/std");
var _constants = require("../../common/constants");
var _non_serializable_instances = require("../reducers/non_serializable_instances");
var _map_selectors = require("../selectors/map_selectors");
var _map_action_constants = require("./map_action_constants");
var _data_request_actions = require("./data_request_actions");
var _layer_actions = require("./layer_actions");
var _vector_layer = require("../classes/layers/vector_layer");
var _ui_actions = require("./ui_actions");
var _geo_tile_utils = require("../classes/util/geo_tile_utils");
var _kibana_services = require("../kibana_services");
var _ui_selectors = require("../selectors/ui_selectors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function setMapInitError(errorMessage) {
  return {
    type: _map_action_constants.SET_MAP_INIT_ERROR,
    errorMessage
  };
}
function setMapSettings(settings) {
  return {
    type: _map_action_constants.SET_MAP_SETTINGS,
    settings
  };
}
function rollbackMapSettings() {
  return {
    type: _map_action_constants.ROLLBACK_MAP_SETTINGS
  };
}
function trackMapSettings() {
  return {
    type: _map_action_constants.TRACK_MAP_SETTINGS
  };
}
function updateMapSetting(settingKey, settingValue) {
  return dispatch => {
    dispatch({
      type: _map_action_constants.UPDATE_MAP_SETTING,
      settingKey,
      settingValue
    });
    if (settingKey === 'autoFitToDataBounds' && settingValue === true) {
      dispatch((0, _data_request_actions.autoFitToBounds)());
    }
  };
}
function updateCustomIcons(customIcons) {
  return {
    type: _map_action_constants.UPDATE_MAP_SETTING,
    settingKey: 'customIcons',
    settingValue: customIcons
  };
}
function deleteCustomIcon(value) {
  return async (dispatch, getState) => {
    const layersContainingCustomIcon = (0, _map_selectors.getLayerList)(getState()).filter(layer => {
      const style = layer.getCurrentStyle();
      if (!style || style.getType() !== _constants.LAYER_STYLE_TYPE.VECTOR) {
        return false;
      }
      return style.isUsingCustomIcon(value);
    });
    if (layersContainingCustomIcon.length > 0) {
      const layerList = await (0, _std.asyncMap)(layersContainingCustomIcon, async layer => {
        return await layer.getDisplayName();
      });
      (0, _kibana_services.getToasts)().addWarning(_i18n.i18n.translate('xpack.maps.mapActions.deleteCustomIconWarning', {
        defaultMessage: `Unable to delete icon. The icon is in use by the {count, plural, one {layer} other {layers}}: {layerNames}`,
        values: {
          count: layerList.length,
          layerNames: layerList.join(', ')
        }
      }));
    } else {
      const newIcons = getState().map.settings.customIcons.filter(({
        symbolId
      }) => symbolId !== value);
      dispatch(updateMapSetting('customIcons', newIcons));
    }
  };
}
function mapReady() {
  return (dispatch, getState) => {
    dispatch({
      type: _map_action_constants.MAP_READY
    });
    const waitingForMapReadyLayerList = (0, _map_selectors.getWaitingForMapReadyLayerListRaw)(getState());
    if ((0, _map_selectors.getMapSettings)(getState()).initialLocation === _constants.INITIAL_LOCATION.AUTO_FIT_TO_BOUNDS) {
      waitingForMapReadyLayerList.forEach(layerDescriptor => {
        dispatch((0, _layer_actions.addLayerWithoutDataSync)(layerDescriptor));
      });
      dispatch((0, _data_request_actions.autoFitToBounds)());
    } else {
      waitingForMapReadyLayerList.forEach(layerDescriptor => {
        dispatch((0, _layer_actions.addLayer)(layerDescriptor));
      });
    }

    // clear waiting list after transfer to avoid state condition
    // where waiting list is empty and layers have not been added to layerList
    dispatch({
      type: _map_action_constants.CLEAR_WAITING_FOR_MAP_READY_LAYER_LIST
    });
  };
}
function mapDestroyed() {
  return {
    type: _map_action_constants.MAP_DESTROYED
  };
}
function mapExtentChanged(mapExtentState) {
  return (dispatch, getState) => {
    const {
      extent,
      zoom: nextZoom
    } = mapExtentState;
    const {
      buffer: prevBuffer,
      zoom: prevZoom
    } = (0, _map_selectors.getDataFilters)(getState());
    let doesPrevBufferContainNextExtent = true;
    if (prevBuffer) {
      const bufferGeometry = (0, _bboxPolygon.default)([prevBuffer.minLon, prevBuffer.minLat, prevBuffer.maxLon, prevBuffer.maxLat]);
      const extentGeometry = (0, _bboxPolygon.default)([extent.minLon, extent.minLat, extent.maxLon, extent.maxLat]);
      doesPrevBufferContainNextExtent = (0, _booleanContains.default)(bufferGeometry, extentGeometry);
    }
    const requiresNewBuffer = !prevBuffer || !doesPrevBufferContainNextExtent || prevZoom !== nextZoom;
    if (requiresNewBuffer) {
      (0, _non_serializable_instances.getInspectorAdapters)(getState()).vectorTiles.setTiles((0, _geo_tile_utils.getTilesForExtent)(nextZoom, extent));
    }
    dispatch({
      type: _map_action_constants.MAP_EXTENT_CHANGED,
      mapViewContext: {
        ...mapExtentState,
        buffer: requiresNewBuffer ? (0, _geo_tile_utils.expandToTileBoundaries)(extent, Math.ceil(nextZoom)) : prevBuffer
      }
    });
    dispatch((0, _data_request_actions.syncDataForAllLayers)(false));
  };
}
function setMouseCoordinates({
  lat,
  lon
}) {
  let safeLon = lon;
  if (lon > 180) {
    const overlapWestOfDateLine = lon - 180;
    safeLon = -180 + overlapWestOfDateLine;
  } else if (lon < -180) {
    const overlapEastOfDateLine = Math.abs(lon) - 180;
    safeLon = 180 - overlapEastOfDateLine;
  }
  return {
    type: _map_action_constants.SET_MOUSE_COORDINATES,
    lat,
    lon: safeLon
  };
}
function clearMouseCoordinates() {
  return {
    type: _map_action_constants.CLEAR_MOUSE_COORDINATES
  };
}
function setGotoWithCenter({
  lat,
  lon,
  zoom
}) {
  return {
    type: _map_action_constants.SET_GOTO,
    center: {
      lat,
      lon,
      zoom
    }
  };
}
function clearGoto() {
  return {
    type: _map_action_constants.CLEAR_GOTO
  };
}
function setQuery({
  query,
  timeFilters,
  timeslice,
  filters,
  forceRefresh = false,
  searchSessionId,
  searchSessionMapBuffer,
  clearTimeslice
}) {
  return async (dispatch, getState) => {
    const prevQuery = (0, _map_selectors.getQuery)(getState());
    const prevTimeFilters = (0, _map_selectors.getTimeFilters)(getState());
    function getNextTimeslice() {
      if (clearTimeslice || timeFilters !== undefined && !_lodash.default.isEqual(timeFilters, prevTimeFilters)) {
        return undefined;
      }
      return timeslice ? timeslice : (0, _map_selectors.getTimeslice)(getState());
    }
    const nextQueryContext = {
      timeFilters: timeFilters ? timeFilters : prevTimeFilters,
      timeslice: getNextTimeslice(),
      query: query ? query : prevQuery,
      filters: filters ? filters : (0, _map_selectors.getFilters)(getState()),
      searchSessionId: searchSessionId ? searchSessionId : (0, _map_selectors.getSearchSessionId)(getState()),
      searchSessionMapBuffer
    };
    const prevQueryContext = {
      timeFilters: prevTimeFilters,
      timeslice: (0, _map_selectors.getTimeslice)(getState()),
      query: prevQuery,
      filters: (0, _map_selectors.getFilters)(getState()),
      searchSessionId: (0, _map_selectors.getSearchSessionId)(getState()),
      searchSessionMapBuffer: (0, _map_selectors.getSearchSessionMapBuffer)(getState())
    };
    if (!forceRefresh && _lodash.default.isEqual(nextQueryContext, prevQueryContext)) {
      // do nothing if query context has not changed
      return;
    }
    dispatch({
      type: _map_action_constants.SET_QUERY,
      ...nextQueryContext
    });
    if ((0, _map_selectors.getMapSettings)(getState()).autoFitToDataBounds) {
      dispatch((0, _data_request_actions.autoFitToBounds)());
    } else {
      await dispatch((0, _data_request_actions.syncDataForAllLayers)(forceRefresh));
    }
  };
}
function setEmbeddableSearchContext({
  query,
  filters
}) {
  return {
    type: _map_action_constants.SET_EMBEDDABLE_SEARCH_CONTEXT,
    embeddableSearchContext: {
      filters,
      query
    }
  };
}
function setExecutionContext(executionContext) {
  return {
    type: _map_action_constants.SET_EXECUTION_CONTEXT,
    executionContext
  };
}
function updateDrawState(drawState) {
  return dispatch => {
    if (drawState !== null) {
      dispatch({
        type: _map_action_constants.SET_OPEN_TOOLTIPS,
        openTooltips: []
      }); // tooltips just get in the way
    }
    dispatch({
      type: _map_action_constants.UPDATE_DRAW_STATE,
      drawState
    });
  };
}
function updateEditShape(shapeToDraw) {
  return (dispatch, getState) => {
    const editState = (0, _map_selectors.getEditState)(getState());
    if (!editState) {
      return;
    }
    dispatch({
      type: _map_action_constants.UPDATE_EDIT_STATE,
      editState: {
        ...editState,
        drawShape: shapeToDraw
      }
    });
    if (shapeToDraw !== _constants.DRAW_SHAPE.DELETE) {
      dispatch((0, _ui_actions.clearDeletedFeatureIds)());
    }
  };
}
function setEditLayerToSelectedLayer() {
  return async (dispatch, getState) => {
    const layerId = (0, _map_selectors.getSelectedLayerId)(getState());
    if (!layerId) {
      return;
    }
    dispatch(updateEditLayer(layerId));
  };
}
function updateEditLayer(layerId) {
  return dispatch => {
    if (layerId !== null) {
      dispatch({
        type: _map_action_constants.SET_OPEN_TOOLTIPS,
        openTooltips: []
      });
    }
    dispatch({
      type: _ui_actions.SET_DRAW_MODE,
      drawMode: _constants.DRAW_MODE.NONE
    });
    dispatch({
      type: _map_action_constants.UPDATE_EDIT_STATE,
      editState: layerId ? {
        layerId
      } : undefined
    });
    dispatch((0, _data_request_actions.syncDataForLayerId)(layerId, false));
  };
}
function addNewFeatureToIndex(geometries) {
  return async (dispatch, getState) => {
    const editState = (0, _map_selectors.getEditState)(getState());
    const layerId = editState ? editState.layerId : undefined;
    if (!layerId) {
      return;
    }
    const layer = (0, _map_selectors.getLayerById)(layerId, getState());
    if (!layer || !(0, _vector_layer.hasVectorLayerMethod)(layer, 'addFeature')) {
      return;
    }
    try {
      dispatch(updateEditShape(_constants.DRAW_SHAPE.WAIT));
      await (0, _std.asyncForEach)(geometries, async geometry => {
        await layer.addFeature(geometry);
      });
      await dispatch((0, _data_request_actions.syncDataForLayerDueToDrawing)(layer));
    } catch (e) {
      (0, _kibana_services.getToasts)().addError(e, {
        title: _i18n.i18n.translate('xpack.maps.mapActions.addFeatureError', {
          defaultMessage: `Unable to add feature to index.`
        })
      });
    }
    dispatch(updateEditShape(_constants.DRAW_SHAPE.SIMPLE_SELECT));
  };
}
function deleteFeatureFromIndex(featureId) {
  return async (dispatch, getState) => {
    // There is a race condition where users can click on a previously deleted feature before layer has re-rendered after feature delete.
    // Check ensures delete requests for previously deleted features are aborted.
    if ((0, _ui_selectors.getDeletedFeatureIds)(getState()).includes(featureId)) {
      return;
    }
    const editState = (0, _map_selectors.getEditState)(getState());
    const layerId = editState ? editState.layerId : undefined;
    if (!layerId) {
      return;
    }
    const layer = (0, _map_selectors.getLayerById)(layerId, getState());
    if (!layer || !(0, _vector_layer.hasVectorLayerMethod)(layer, 'deleteFeature')) {
      return;
    }
    try {
      dispatch(updateEditShape(_constants.DRAW_SHAPE.WAIT));
      await layer.deleteFeature(featureId);
      dispatch((0, _ui_actions.pushDeletedFeatureId)(featureId));
      await dispatch((0, _data_request_actions.syncDataForLayerDueToDrawing)(layer));
    } catch (e) {
      (0, _kibana_services.getToasts)().addError(e, {
        title: _i18n.i18n.translate('xpack.maps.mapActions.removeFeatureError', {
          defaultMessage: `Unable to remove feature from index.`
        })
      });
    }
    dispatch(updateEditShape(_constants.DRAW_SHAPE.DELETE));
  };
}