"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Pipeline = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// This config template is presented to the user for the 'new pipeline' workflow
const emptyPipeline = 'input {\n' + '}\n' + 'filter {\n' + '}\n' + 'output {\n' + '}';

// Should be consistent with https://www.elastic.co/guide/en/logstash/current/logstash-settings-file.html
const settingsDefaults = {
  'pipeline.workers': null,
  // Defaults to number of CPU cores
  'pipeline.batch.size': 125,
  'pipeline.batch.delay': 50,
  'queue.type': 'memory',
  'queue.max_bytes.number': 1,
  'queue.max_bytes.units': 'gb',
  'queue.checkpoint.writes': 1024
};
class Pipeline {
  /**
   * Represents the pipeline for the client side editing/creating workflow
   * @param {import('./props').Props} props}
   */
  constructor(props = undefined) {
    (0, _defineProperty2.default)(this, "isEqualTo", otherPipeline => {
      // We need to create a POJO copies because isEqual would return false
      // because of property getters
      const cleanPipeline = {
        ...this
      };
      const cleanOtherPipeline = {
        ...otherPipeline
      };
      return (0, _lodash.isEqual)(cleanPipeline, cleanOtherPipeline);
    });
    this.id = (0, _lodash.get)(props, 'id');
    this.description = (0, _lodash.get)(props, 'description', '');
    this.pipeline = (0, _lodash.get)(props, 'pipeline', emptyPipeline);
    this.username = (0, _lodash.get)(props, 'username');
    this.settings = (0, _lodash.defaultsDeep)((0, _lodash.get)(props, 'settings', {}), settingsDefaults);
  }
  get clone() {
    return new Pipeline({
      ...(0, _lodash.omit)(this, ['id', 'username'])
    });
  }
  get upstreamJSON() {
    const settings = this.settings;
    const maxBytesNumber = (0, _lodash.get)(settings, 'queue.max_bytes.number');
    const maxBytesUnits = (0, _lodash.get)(settings, 'queue.max_bytes.units');
    const upstreamSettings = {
      ...settings
    };
    if (maxBytesNumber && maxBytesUnits) {
      delete upstreamSettings['queue.max_bytes.number'];
      delete upstreamSettings['queue.max_bytes.units'];
      upstreamSettings['queue.max_bytes'] = `${maxBytesNumber}${maxBytesUnits}`;
    }
    return {
      description: this.description,
      pipeline: this.pipeline,
      username: this.username,
      settings: upstreamSettings
    };
  }
  static fromUpstreamJSON(pipeline) {
    const settings = pipeline.settings;
    const maxBytesStr = (0, _lodash.get)(settings, 'queue.max_bytes', '');
    const maxBytesParts = maxBytesStr.match(/(\d+)(\w+)/);
    if (Array.isArray(maxBytesParts) && maxBytesParts.length === 3) {
      const maxBytesNumber = maxBytesParts[1];
      const maxBytesUnits = maxBytesParts[2];
      if (maxBytesNumber && maxBytesUnits) {
        delete settings['queue.max_bytes'];
        settings['queue.max_bytes.number'] = parseInt(maxBytesNumber);
        settings['queue.max_bytes.units'] = maxBytesUnits;
      }
    }
    return new Pipeline({
      id: pipeline.id,
      description: pipeline.description,
      pipeline: pipeline.pipeline,
      username: pipeline.username,
      settings
    });
  }
}
exports.Pipeline = Pipeline;