"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createGetLogErrorRateTimeseries = createGetLogErrorRateTimeseries;
var _utils = require("../../utils");
var _es_fields = require("../../es_fields");
var _es_queries = require("../../utils/es_queries");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getLogErrorsAggregation = () => ({
  terms: {
    field: _es_fields.LOG_LEVEL,
    include: ['error', 'ERROR']
  }
});
const getErrorLogLevelErrorsAggregation = () => ({
  terms: {
    field: _es_fields.ERROR_LOG_LEVEL,
    include: ['error', 'ERROR']
  }
});
function createGetLogErrorRateTimeseries() {
  return async ({
    esClient,
    identifyingMetadata,
    serviceNames,
    timeFrom,
    timeTo,
    kuery,
    serviceEnvironmentQuery = []
  }) => {
    const intervalString = (0, _utils.getBucketSizeFromTimeRangeAndBucketCount)(timeFrom, timeTo, 50);

    // Note: Please keep the formula in `metricsFormulasMap` up to date with the query!

    const esResponse = await esClient.search({
      index: 'logs-*-*',
      size: 0,
      query: {
        bool: {
          filter: [...(0, _es_queries.kqlQuery)(kuery), {
            terms: {
              [identifyingMetadata]: serviceNames
            }
          }, ...serviceEnvironmentQuery, {
            range: {
              ['@timestamp']: {
                gte: timeFrom,
                lte: timeTo,
                format: 'epoch_millis'
              }
            }
          }]
        }
      },
      aggs: {
        services: {
          terms: {
            field: identifyingMetadata
          },
          aggs: {
            timeseries: {
              date_histogram: {
                field: '@timestamp',
                fixed_interval: `${intervalString}s`,
                min_doc_count: 0,
                extended_bounds: {
                  min: timeFrom,
                  max: timeTo
                }
              },
              aggs: {
                logErrors: getLogErrorsAggregation(),
                errorLogLevelErrors: getErrorLogLevelErrorsAggregation()
              }
            }
          }
        }
      }
    });
    const aggregations = esResponse.aggregations;
    const buckets = aggregations === null || aggregations === void 0 ? void 0 : aggregations.services.buckets;
    return buckets ? buckets.reduce((acc, bucket) => {
      const timeseries = bucket.timeseries.buckets.map(timeseriesBucket => {
        var _timeseriesBucket$log, _timeseriesBucket$err, _timeseriesBucket$err2;
        const logErrorCount = ((_timeseriesBucket$log = timeseriesBucket.logErrors.buckets[0]) === null || _timeseriesBucket$log === void 0 ? void 0 : _timeseriesBucket$log.doc_count) || 0;
        const errorLogLevelErrorsCount = ((_timeseriesBucket$err = timeseriesBucket.errorLogLevelErrors) === null || _timeseriesBucket$err === void 0 ? void 0 : (_timeseriesBucket$err2 = _timeseriesBucket$err.buckets[0]) === null || _timeseriesBucket$err2 === void 0 ? void 0 : _timeseriesBucket$err2.doc_count) || 0;
        const totalErrorsCount = logErrorCount + errorLogLevelErrorsCount;
        return {
          x: timeseriesBucket.key,
          y: totalErrorsCount / (intervalString / 60)
        };
      });
      return {
        ...acc,
        [bucket.key]: timeseries
      };
    }, {}) : {};
  };
}