"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findExceptionListItemRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _zodHelpers = require("@kbn/zod-helpers");
var _api = require("@kbn/securitysolution-exceptions-common/api");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const findExceptionListItemRoute = router => {
  router.versioned.get({
    access: 'public',
    path: `${_securitysolutionListConstants.EXCEPTION_LIST_ITEM_URL}/_find`,
    security: {
      authz: {
        requiredPrivileges: ['lists-read']
      }
    }
  }).addVersion({
    validate: {
      request: {
        query: (0, _zodHelpers.buildRouteValidationWithZod)(_api.FindExceptionListItemsRequestQuery)
      }
    },
    version: '2023-10-31'
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    try {
      const exceptionLists = await (0, _utils.getExceptionListClient)(context);
      const {
        filter,
        list_id: listId,
        namespace_type: namespaceType,
        page,
        per_page: perPage,
        search,
        sort_field: sortField,
        sort_order: sortOrder
      } = request.query;
      if (listId.length !== namespaceType.length) {
        return siemResponse.error({
          body: `list_id and namespace_id need to have the same comma separated number of values. Expected list_id length: ${listId.length} to equal namespace_type length: ${namespaceType.length}`,
          statusCode: 400
        });
      } else if (listId.length !== filter.length && filter.length !== 0) {
        return siemResponse.error({
          body: `list_id and filter need to have the same comma separated number of values. Expected list_id length: ${listId.length} to equal filter length: ${filter.length}`,
          statusCode: 400
        });
      }
      const exceptionListItems = await exceptionLists.findExceptionListsItem({
        filter,
        listId,
        namespaceType,
        page,
        perPage,
        pit: undefined,
        search,
        searchAfter: undefined,
        sortField,
        sortOrder
      });
      if (exceptionListItems == null) {
        return siemResponse.error({
          body: `exception list id: "${listId}" does not exist`,
          statusCode: 404
        });
      }
      return response.ok({
        body: _api.FindExceptionListItemsResponse.parse(exceptionListItems)
      });
    } catch (err) {
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return siemResponse.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.findExceptionListItemRoute = findExceptionListItemRoute;