"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.licensingSessionStorageKey = exports.LicensingPlugin = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _license_update = require("../common/license_update");
var _license = require("../common/license");
var _expired_banner = require("./expired_banner");
var _services = require("./services");
var _register_analytics_context_provider = require("../common/register_analytics_context_provider");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const licensingSessionStorageKey = exports.licensingSessionStorageKey = 'xpack.licensing';

/**
 * @public
 * A plugin for fetching, refreshing, and receiving information about the license for the
 * current Kibana instance.
 */
class LicensingPlugin {
  constructor(_context, storage = sessionStorage) {
    /**
     * Used as a flag to halt all other plugin observables.
     */
    (0, _defineProperty2.default)(this, "stop$", new _rxjs.Subject());
    /**
     * A function to execute once the plugin's HTTP interceptor needs to stop listening.
     */
    (0, _defineProperty2.default)(this, "removeInterceptor", void 0);
    (0, _defineProperty2.default)(this, "internalSubscription", void 0);
    (0, _defineProperty2.default)(this, "isLicenseExpirationBannerShown", false);
    (0, _defineProperty2.default)(this, "infoEndpoint", '/api/licensing/info');
    (0, _defineProperty2.default)(this, "coreStart", void 0);
    (0, _defineProperty2.default)(this, "prevSignature", void 0);
    (0, _defineProperty2.default)(this, "refresh", void 0);
    (0, _defineProperty2.default)(this, "license$", void 0);
    (0, _defineProperty2.default)(this, "featureUsage", new _services.FeatureUsageService());
    (0, _defineProperty2.default)(this, "fetchLicense", async core => {
      try {
        const response = await core.http.get({
          path: this.infoEndpoint,
          asSystemRequest: true
        });
        return new _license.License({
          license: response.license,
          features: response.features,
          signature: response.signature
        });
      } catch (error) {
        return new _license.License({
          error: error.message,
          signature: ''
        });
      }
    });
    this.storage = storage;
  }

  /**
   * Fetch the objectified license and signature from storage.
   */
  getSaved() {
    const raw = this.storage.getItem(licensingSessionStorageKey);
    if (!raw) return;
    return _license.License.fromJSON(JSON.parse(raw));
  }

  /**
   * Store the given license and signature in storage.
   */
  save(license) {
    this.storage.setItem(licensingSessionStorageKey, JSON.stringify(license));
  }

  /**
   * Clear license and signature information from storage.
   */
  removeSaved() {
    this.storage.removeItem(licensingSessionStorageKey);
  }
  setup(core) {
    const signatureUpdated$ = new _rxjs.Subject();
    const {
      license$,
      refreshManually
    } = (0, _license_update.createLicenseUpdate)(signatureUpdated$, this.stop$, () => this.fetchLicense(core), this.getSaved());
    (0, _register_analytics_context_provider.registerAnalyticsContextProvider)(core.analytics, license$);
    this.internalSubscription = license$.subscribe(license => {
      if (license.isAvailable) {
        this.prevSignature = license.signature;
        this.save(license);
      } else {
        this.prevSignature = undefined;
        // Prevent reusing stale license if the fetch operation fails
        this.removeSaved();
      }
      if (license.status === 'expired' && !this.isLicenseExpirationBannerShown && this.coreStart) {
        this.isLicenseExpirationBannerShown = true;
        this.showExpiredBanner(license);
      }
    });
    this.removeInterceptor = core.http.intercept({
      response: async httpResponse => {
        // we don't track license as anon users do not have one.
        if (core.http.anonymousPaths.isAnonymous(window.location.pathname)) return httpResponse;
        if (httpResponse.response) {
          const signatureHeader = httpResponse.response.headers.get('kbn-license-sig');
          if (typeof signatureHeader === 'string' && this.prevSignature !== signatureHeader) {
            if (!httpResponse.request.url.includes(this.infoEndpoint)) {
              signatureUpdated$.next();
            }
          }
        }
        return httpResponse;
      }
    });
    this.refresh = refreshManually;
    this.license$ = license$;
    return {
      refresh: refreshManually,
      license$,
      featureUsage: this.featureUsage.setup()
    };
  }
  start(core) {
    this.coreStart = core;
    if (!this.refresh || !this.license$) {
      throw new Error('Setup has not been completed');
    }
    return {
      refresh: this.refresh,
      getLicense: async () => await (0, _rxjs.firstValueFrom)(this.license$),
      license$: this.license$,
      featureUsage: this.featureUsage.start({
        http: core.http
      })
    };
  }
  stop() {
    this.stop$.next();
    this.stop$.complete();
    if (this.removeInterceptor !== undefined) {
      this.removeInterceptor();
    }
    if (this.internalSubscription !== undefined) {
      this.internalSubscription.unsubscribe();
      this.internalSubscription = undefined;
    }
  }
  showExpiredBanner(license) {
    const coreStart = this.coreStart;
    const uploadUrl = coreStart.http.basePath.prepend('/app/management/stack/license_management/upload_license');
    coreStart.overlays.banners.add((0, _expired_banner.mountExpiredBanner)({
      type: license.type,
      uploadUrl,
      ...coreStart
    }));
  }
}
exports.LicensingPlugin = LicensingPlugin;