"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.LensStorage = void 0;
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _contentManagementUtils = require("@kbn/content-management-utils");
var _constants = require("../../common/constants");
var _services = require("./services");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const searchArgsToSOFindOptions = args => {
  var _ref;
  const {
    query,
    contentTypeId,
    options
  } = args;

  // Shared schema type allows string here, need to align manually
  const searchFields = (_ref = options !== null && options !== void 0 && options.searchFields ? Array.isArray(options.searchFields) ? options.searchFields : [options.searchFields] : null) !== null && _ref !== void 0 ? _ref : ['title^3', 'description'];
  return {
    type: contentTypeId,
    search: query.text,
    perPage: query.limit,
    page: query.cursor ? +query.cursor : undefined,
    defaultSearchOperator: 'AND',
    ...options,
    ...(0, _contentManagementUtils.tagsToFindOptions)(query.tags),
    searchFields
  };
};
class LensStorage extends _contentManagementUtils.SOContentStorage {
  constructor(params) {
    super({
      savedObjectType: _constants.LENS_CONTENT_TYPE,
      cmServicesDefinition: _services.servicesDefinitions,
      searchArgsToSOFindOptions,
      enableMSearch: true,
      allowedSavedObjectAttributes: ['title', 'description', 'visualizationType', 'version', 'state'],
      logger: params.logger,
      throwOnResultValidationError: params.throwOnResultValidationError
    });
    this.mSearch.toItemResult = (ctx, savedObject) => {
      var _attributes$descripti;
      const transforms = ctx.utils.getTransforms(_services.servicesDefinitions);
      const {
        attributes,
        ...rest
      } = this.savedObjectToItem(
      // TODO: Fix this typing, this is not true we don't necessarily have all the Lens attributes
      savedObject);
      const commonContentItem = {
        updatedAt: '',
        // type misalignment
        ...rest,
        attributes: {
          ...attributes,
          // temporarily pass all attributes for SO finder usages
          title: attributes.title,
          description: (_attributes$descripti = attributes.description) !== null && _attributes$descripti !== void 0 ? _attributes$descripti : ''
        }
      };
      const validationError = transforms.mSearch.out.result.validate(commonContentItem);
      if (validationError) {
        if (this.throwOnResultValidationError) {
          throw _boom.default.badRequest(`Invalid response. ${validationError.message}`);
        } else {
          this.logger.warn(`Invalid response. ${validationError.message}`);
        }
      }

      // TODO: Fix this typing, this expects a full item attributes but only has title and description
      return commonContentItem;
    };
  }
  async get(ctx, id) {
    var _response$item$attrib;
    const soClient = await LensStorage.getSOClientFromRequest(ctx);

    // Save data in DB
    const {
      saved_object: savedObject,
      alias_purpose: aliasPurpose,
      alias_target_id: aliasTargetId,
      outcome
    } = await soClient.resolve(_constants.LENS_CONTENT_TYPE, id);
    const response = {
      item: this.savedObjectToItem(savedObject),
      meta: {
        aliasPurpose,
        aliasTargetId,
        outcome
      }
    };
    const itemVersion = (_response$item$attrib = response.item.attributes.version) !== null && _response$item$attrib !== void 0 ? _response$item$attrib : 0;
    const transforms = ctx.utils.getTransforms(_services.servicesDefinitions);

    // transform item from given version to latest version
    const {
      value,
      error: resultError
    } = transforms.get.out.result.down(response, itemVersion, {
      validate: true
    } // validate initial SO Item
    );
    if (resultError) {
      throw _boom.default.badRequest(`Transform error. ${resultError.message}`);
    }
    const validationError = transforms.get.out.result.validate(value);
    if (validationError) {
      if (this.throwOnResultValidationError) {
        throw _boom.default.badRequest(`Invalid response. ${validationError.message}`);
      } else {
        this.logger.warn(`Invalid response. ${validationError.message}`);
      }
    }
    return value;
  }
  async create(ctx, data, options = {}) {
    var _data$version;
    const transforms = ctx.utils.getTransforms(_services.servicesDefinitions);
    const soClient = await _contentManagementUtils.SOContentStorage.getSOClientFromRequest(ctx);
    const itemVersion = (_data$version = data.version) !== null && _data$version !== void 0 ? _data$version : 0; // Check that this always has a version

    // transform data from given version to latest version
    const {
      value: dataToLatest,
      error: dataError
    } = transforms.create.in.data.down(data, itemVersion);
    if (dataError) {
      throw _boom.default.badRequest(`Invalid data. ${dataError.message}`);
    }

    // transform options from given version to latest version
    const {
      value: optionsToLatest,
      error: optionsError
    } = transforms.create.in.options.down(options, itemVersion);
    if (optionsError) {
      throw _boom.default.badRequest(`Invalid options. ${optionsError.message}`);
    }
    const createOptions = this.createArgsToSoCreateOptions(optionsToLatest !== null && optionsToLatest !== void 0 ? optionsToLatest : {});

    // Save data in DB
    const savedObject = await soClient.create(_constants.LENS_CONTENT_TYPE, dataToLatest, createOptions);
    const result = {
      item: this.savedObjectToItem(savedObject)
    };
    const validationError = transforms.create.out.result.validate(result);
    if (validationError) {
      if (this.throwOnResultValidationError) {
        throw _boom.default.badRequest(`Invalid response. ${validationError.message}`);
      } else {
        this.logger.warn(`Invalid response. ${validationError.message}`);
      }
    }

    // transform result from latest version to request version
    const {
      value,
      error: resultError
    } = transforms.create.out.result.down(result, 'latest', {
      validate: false
    } // validation is done above
    );
    if (resultError) {
      throw _boom.default.badRequest(`Invalid response. ${resultError.message}`);
    }
    return value;
  }

  /**
   * Lens requires a custom update function because of https://github.com/elastic/kibana/issues/160116
   * where a forced create with overwrite flag is used instead of regular update
   */
  async update(ctx, id, data, options) {
    var _data$version2;
    const {
      utils: {
        getTransforms
      },
      version: {
        request: requestVersion
      }
    } = ctx;
    const transforms = getTransforms(_services.servicesDefinitions, requestVersion);
    const itemVersion = (_data$version2 = data.version) !== null && _data$version2 !== void 0 ? _data$version2 : 0; // Check that this always has a version

    // transform data from given version to latest version
    const {
      value: dataToLatest,
      error: dataError
    } = transforms.update.in.data.down(data, itemVersion);
    if (dataError) {
      throw _boom.default.badRequest(`Invalid data. ${dataError.message}`);
    }

    // transform options from given version to latest version
    const {
      value: optionsToLatest,
      error: optionsError
    } = transforms.update.in.options.down(options, itemVersion);
    if (optionsError) {
      throw _boom.default.badRequest(`Invalid options. ${optionsError.message}`);
    }
    const soClient = await LensStorage.getSOClientFromRequest(ctx);

    // since we use create below this call is meant to throw if SO id not found
    await soClient.get(_constants.LENS_CONTENT_TYPE, id);
    const savedObject = await soClient.create(_constants.LENS_CONTENT_TYPE, dataToLatest, {
      id,
      overwrite: true,
      ...optionsToLatest
    });
    const result = {
      item: this.savedObjectToItem(savedObject)
    };
    const validationError = transforms.update.out.result.validate(result);
    if (validationError) {
      if (this.throwOnResultValidationError) {
        throw _boom.default.badRequest(`Invalid response. ${validationError.message}`);
      } else {
        this.logger.warn(`Invalid response. ${validationError.message}`);
      }
    }

    // transform result from latest version to request version
    const {
      value,
      error: resultError
    } = transforms.update.out.result.down(result, 'latest', {
      validate: false
    } // validation is done above
    );
    if (resultError) {
      throw _boom.default.badRequest(`Invalid response. ${resultError.message}`);
    }
    return value;
  }
  async search(ctx, query, options = {}) {
    const transforms = ctx.utils.getTransforms(_services.servicesDefinitions);
    const soClient = await _contentManagementUtils.SOContentStorage.getSOClientFromRequest(ctx);
    const requestVersion = 'latest'; // this should eventually come from the request when there is a v2

    // Validate and UP transform the options
    const {
      value: optionsToLatest,
      error: optionsError
    } = transforms.search.in.options.up(options, requestVersion);
    if (optionsError) {
      throw _boom.default.badRequest(`Invalid payload. ${optionsError.message}`);
    }
    const soQuery = this.searchArgsToSOFindOptions({
      contentTypeId: _constants.LENS_CONTENT_TYPE,
      query,
      options: optionsToLatest
    });

    // Execute the query in the DB
    const soResponse = await soClient.find(soQuery);
    const items = soResponse.saved_objects.map(so => this.savedObjectToItem(so));
    const transformedItems = items.map(item => {
      var _item$attributes$vers;
      // transform item from given version to latest version
      const {
        value: transformedItem,
        error: itemError
      } = transforms.search.out.result.down(item, (_item$attributes$vers = item.attributes.version) !== null && _item$attributes$vers !== void 0 ? _item$attributes$vers : 0, {
        validate: false
      } // validation is done after transform below
      );
      if (itemError) {
        throw _boom.default.badRequest(`Transform error. ${itemError.message}`);
      }
      return transformedItem;
    });
    const response = {
      hits: transformedItems,
      pagination: {
        total: soResponse.total
      }
    };
    const validationError = transforms.search.out.result.validate(response);
    if (validationError) {
      if (this.throwOnResultValidationError) {
        throw _boom.default.badRequest(`Invalid response. ${validationError.message}`);
      } else {
        this.logger.warn(`Invalid response. ${validationError.message}`);
      }
    }
    return response;
  }
}
exports.LensStorage = LensStorage;