"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.filterByVisibleOperation = filterByVisibleOperation;
exports.findMathNodes = findMathNodes;
exports.findVariables = findVariables;
exports.getOperationParams = getOperationParams;
exports.getValueOrName = getValueOrName;
exports.groupArgsByType = groupArgsByType;
exports.isMathNode = isMathNode;
exports.mergeWithGlobalFilters = mergeWithGlobalFilters;
exports.unquotedStringRegex = void 0;
var _lodash = require("lodash");
var _lensFormulaDocs = require("@kbn/lens-formula-docs");
var _utils = require("../../../../../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const unquotedStringRegex = exports.unquotedStringRegex = /[^0-9A-Za-z._@\[\]/]/;
function groupArgsByType(args) {
  const {
    namedArgument,
    variable,
    function: functions
  } = (0, _lodash.groupBy)(args, arg => {
    return (0, _lodash.isObject)(arg) ? arg.type : 'variable';
  });
  // better naming
  return {
    namedArguments: namedArgument || [],
    variables: variable || [],
    functions: functions || []
  };
}
function getValueOrName(node) {
  if (!(0, _lodash.isObject)(node)) {
    return node;
  }
  if (node.type !== 'function') {
    return node.value;
  }
  return node.name;
}
function mergeWithGlobalFilters(operation, mappedParams, globalFilter, globalReducedTimeRange) {
  if (globalFilter && operation.filterable) {
    const languageKey = 'kql' in mappedParams ? 'kql' : 'lucene';
    if (mappedParams[languageKey]) {
      // ignore the initial empty string case
      if (globalFilter.query) {
        mappedParams[languageKey] = `(${globalFilter.query}) AND (${mappedParams[languageKey]})`;
      }
    } else {
      const language = globalFilter.language === 'kuery' ? 'kql' : globalFilter.language;
      mappedParams[language] = globalFilter.query;
    }
  }
  // Local definition override the global one
  if (globalReducedTimeRange && operation.canReduceTimeRange && !mappedParams.reducedTimeRange) {
    mappedParams.reducedTimeRange = globalReducedTimeRange;
  }
  return mappedParams;
}
function getOperationParams(operation, params = []) {
  const formalArgs = (operation.operationParams || []).reduce((memo, {
    name,
    type
  }) => {
    memo[name] = type;
    return memo;
  }, {});
  return params.reduce((args, {
    name,
    value
  }) => {
    if (formalArgs[name]) {
      args[name] = value;
    }
    if (operation.filterable && (name === 'kql' || name === 'lucene')) {
      args[name] = value;
    }
    if (operation.shiftable && name === 'shift') {
      args[name] = value;
    }
    if (operation.canReduceTimeRange && name === 'reducedTimeRange') {
      args.reducedTimeRange = value;
    }
    return args;
  }, {});
}
function isMathNode(node) {
  return (0, _lodash.isObject)(node) && node.type === 'function' && _lensFormulaDocs.tinymathFunctions[node.name];
}
function findMathNodes(root) {
  function flattenMathNodes(node) {
    if (!(0, _lodash.isObject)(node) || node.type !== 'function' || !isMathNode(node)) {
      return [];
    }
    return [node, ...node.args.flatMap(flattenMathNodes)].filter(_utils.nonNullable);
  }
  return flattenMathNodes(root);
}

// traverse a tree and find all string leaves
function findVariables(node) {
  if (typeof node === 'string') {
    return [{
      type: 'variable',
      value: node,
      text: node,
      location: {
        min: 0,
        max: 0
      }
    }];
  }
  if (node == null) {
    return [];
  }
  if (typeof node === 'number' || node.type === 'namedArgument') {
    return [];
  }
  if (node.type === 'variable') {
    // leaf node
    return [node];
  }
  return node.args.flatMap(findVariables);
}
function filterByVisibleOperation(operationDefinitionMap) {
  return Object.fromEntries(Object.entries(operationDefinitionMap).filter(([, operation]) => !operation.hidden));
}