"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createCounterFetcher = createCounterFetcher;
exports.transformRawCounter = transformRawCounter;
var _moment = _interopRequireDefault(require("moment"));
var _server = require("@kbn/usage-collection-plugin/server");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function createCounterFetcher(logger, filter, transform) {
  return async ({
    soClient
  }) => {
    const finder = soClient.createPointInTimeFinder({
      type: _server.USAGE_COUNTERS_SAVED_OBJECT_TYPE,
      namespaces: ['*'],
      fields: ['count', 'counterName', 'counterType', 'domainId'],
      filter,
      perPage: 100
    });
    const dailyEvents = [];
    for await (const {
      saved_objects: rawUsageCounters
    } of finder.find()) {
      rawUsageCounters.forEach(raw => {
        try {
          const event = transformRawCounter(raw);
          if (event) {
            dailyEvents.push(event);
          }
        } catch (err) {
          // swallow error; allows sending successfully transformed objects.
          logger.debug('Error collecting usage-counter details: ' + err.message);
        }
      });
    }
    return transform(mergeCounters(dailyEvents));
  };
}
function transformRawCounter(rawCounter) {
  const {
    attributes: {
      domainId,
      counterType,
      counterName,
      count
    },
    updated_at: lastUpdatedAt
  } = rawCounter;
  if (typeof count !== 'number' || count < 1) {
    return;
  }
  const fromTimestamp = (0, _moment.default)(lastUpdatedAt).utc().startOf('day').format();
  return {
    domainId,
    counterType,
    counterName,
    lastUpdatedAt: lastUpdatedAt,
    fromTimestamp,
    total: count
  };
}
function mergeCounters(counters) {
  const mergedCounters = counters.reduce((acc, counter) => {
    const {
      domainId,
      counterType,
      counterName,
      fromTimestamp
    } = counter;
    const key = `${domainId}:${counterType}:${counterName}:${fromTimestamp}`;
    const existingCounter = acc[key];
    if (!existingCounter) {
      acc[key] = counter;
      return acc;
    } else {
      acc[key].total = existingCounter.total + counter.total;
      acc[key].lastUpdatedAt = counter.lastUpdatedAt;
    }
    return acc;
  }, {});
  return Object.values(mergedCounters);
}