"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.KibanaFramework = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _server = require("@kbn/data-plugin/server");
var _common = require("@kbn/spaces-plugin/common");
var _cancel_request_on_abort = require("../../cancel_request_on_abort");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class KibanaFramework {
  constructor(core, config, plugins) {
    (0, _defineProperty2.default)(this, "router", void 0);
    (0, _defineProperty2.default)(this, "plugins", void 0);
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "core", void 0);
    this.router = core.http.createRouter();
    this.plugins = plugins;
    this.core = core;
    this.config = config;
  }
  registerRoute(config, handler) {
    const defaultSecurity = {
      authz: {
        requiredPrivileges: ['infra']
      }
    };
    const routeConfig = {
      path: config.path,
      validate: config.validate,
      /**
       * Supported `options` for each type of request method
       * are a bit different and generic method like this cannot
       * properly ensure type safety. Hence the need to cast
       * using `as ...` below to ensure the route config has
       * the correct options type.
       */
      options: {
        ...config.options
      },
      security: defaultSecurity
    };
    switch (config.method) {
      case 'get':
        this.router.get(routeConfig, handler);
        break;
      case 'post':
        this.router.post(routeConfig, handler);
        break;
      case 'delete':
        this.router.delete(routeConfig, handler);
        break;
      case 'put':
        this.router.put(routeConfig, handler);
        break;
      case 'patch':
        this.router.patch(routeConfig, handler);
        break;
    }
  }
  registerVersionedRoute(config) {
    const defaultSecurity = {
      authz: {
        requiredPrivileges: ['infra']
      }
    };
    const routeConfig = {
      access: config.access,
      path: config.path,
      security: defaultSecurity
    };
    switch (config.method) {
      case 'get':
        return this.router.versioned.get(routeConfig);
      case 'post':
        return this.router.versioned.post(routeConfig);
      case 'delete':
        return this.router.versioned.delete(routeConfig);
      case 'put':
        return this.router.versioned.put(routeConfig);
      case 'patch':
        return this.router.versioned.patch(routeConfig);
      default:
        throw new RangeError(`#registerVersionedRoute: "${config.method}" is not an accepted method`);
    }
  }
  async callWithRequest(requestContext, endpoint, params, request) {
    const {
      elasticsearch,
      uiSettings
    } = await requestContext.core;
    const includeFrozen = await uiSettings.client.get(_server.UI_SETTINGS.SEARCH_INCLUDE_FROZEN);
    if (endpoint === 'msearch') {
      const maxConcurrentShardRequests = await uiSettings.client.get(_server.UI_SETTINGS.COURIER_MAX_CONCURRENT_SHARD_REQUESTS);
      if (maxConcurrentShardRequests > 0) {
        params = {
          ...params,
          max_concurrent_shard_requests: maxConcurrentShardRequests
        };
      }
    }
    function callWrapper({
      makeRequestWithSignal
    }) {
      const controller = new AbortController();
      const promise = makeRequestWithSignal(controller.signal);
      return request ? (0, _cancel_request_on_abort.subscribeToAborted$)(promise, request, controller) : promise;
    }

    // Only set the "ignore_throttled" value (to false) if the Kibana setting
    // for "search:includeFrozen" is true (i.e. don't ignore throttled indices, a triple negative!)
    // More information:
    // - https://github.com/elastic/kibana/issues/113197
    // - https://github.com/elastic/elasticsearch/pull/77479
    //
    // NOTE: these params only need to be spread onto the search and msearch calls below
    const frozenIndicesParams = {};
    if (includeFrozen) {
      frozenIndicesParams.ignore_throttled = false;
    }
    let apiResult;
    switch (endpoint) {
      case 'search':
        apiResult = callWrapper({
          makeRequestWithSignal: signal => elasticsearch.client.asCurrentUser.search({
            ...params,
            ...frozenIndicesParams
          }, {
            signal
          })
        });
        break;
      case 'msearch':
        apiResult = callWrapper({
          makeRequestWithSignal: signal => elasticsearch.client.asCurrentUser.msearch({
            ...params,
            ...frozenIndicesParams
          }, {
            signal
          })
        });
        break;
      case 'indices.existsAlias':
        apiResult = callWrapper({
          makeRequestWithSignal: signal => elasticsearch.client.asCurrentUser.indices.existsAlias({
            ...params
          }, {
            signal
          })
        });
        break;
      case 'indices.getAlias':
        apiResult = callWrapper({
          makeRequestWithSignal: signal => elasticsearch.client.asCurrentUser.indices.getAlias({
            ...params
          }, {
            signal
          })
        });
        break;
      case 'indices.get':
        apiResult = callWrapper({
          makeRequestWithSignal: signal => elasticsearch.client.asCurrentUser.indices.get({
            ...params
          }, {
            signal
          })
        });
        break;
      case 'transport.request':
        apiResult = callWrapper({
          makeRequestWithSignal: signal => elasticsearch.client.asCurrentUser.transport.request({
            ...params
          }, {
            signal
          })
        });
        break;
      case 'ml.getBuckets':
        apiResult = callWrapper({
          makeRequestWithSignal: signal => elasticsearch.client.asCurrentUser.ml.getBuckets({
            ...params
          }, {
            signal
          })
        });
        break;
    }
    return apiResult ? await apiResult : undefined;
  }
  async getIndexPatternsServiceWithRequestContext(requestContext) {
    const {
      savedObjects,
      elasticsearch
    } = await requestContext.core;
    return await this.createIndexPatternsService(savedObjects.client, elasticsearch.client.asCurrentUser);
  }
  async getIndexPatternsService(savedObjectsClient, elasticsearchClient) {
    return await this.createIndexPatternsService(savedObjectsClient, elasticsearchClient);
  }
  async createIndexPatternsService(savedObjectsClient, elasticsearchClient) {
    const [, startPlugins] = await this.core.getStartServices();
    return startPlugins.data.indexPatterns.dataViewsServiceFactory(savedObjectsClient, elasticsearchClient);
  }
  getSpaceId(request) {
    var _this$plugins$spaces$, _this$plugins$spaces, _this$plugins$spaces$2;
    return (_this$plugins$spaces$ = (_this$plugins$spaces = this.plugins.spaces) === null || _this$plugins$spaces === void 0 ? void 0 : (_this$plugins$spaces$2 = _this$plugins$spaces.spacesService) === null || _this$plugins$spaces$2 === void 0 ? void 0 : _this$plugins$spaces$2.getSpaceId(request)) !== null && _this$plugins$spaces$ !== void 0 ? _this$plugins$spaces$ : _common.DEFAULT_SPACE_ID;
  }
  async makeTSVBRequest(requestContext, rawRequest, model, timerange, filters) {
    const {
      getVisData
    } = this.plugins.visTypeTimeseries;
    if (typeof getVisData !== 'function') {
      throw new Error('TSVB is not available');
    }
    const options = {
      timerange,
      panels: [model],
      filters
    };
    return getVisData(requestContext, rawRequest, options);
  }
}
exports.KibanaFramework = KibanaFramework;