"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.applyWaffleMapLayout = applyWaffleMapLayout;
exports.getColumns = getColumns;
exports.getLargestCount = getLargestCount;
exports.getTotalItems = getTotalItems;
var _lodash = require("lodash");
var _type_guards = require("./type_guards");
var _size_of_squares = require("./size_of_squares");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getColumns(n, w = 1, h = 1) {
  const pageRatio = w / h;
  const ratio = pageRatio > 1.2 ? 1.2 : pageRatio;
  const width = Math.ceil(Math.sqrt(n));
  return Math.ceil(width * ratio);
}
function getTotalItems(groups) {
  if (!groups) {
    return 0;
  }
  return groups.reduce((acc, group) => {
    if ((0, _type_guards.isWaffleMapGroupWithGroups)(group)) {
      return group.groups.reduce((total, subGroup) => subGroup.nodes.length + total, acc);
    }
    if ((0, _type_guards.isWaffleMapGroupWithNodes)(group)) {
      return group.nodes.length + acc;
    }
    return acc;
  }, 0);
}
function getLargestCount(groups) {
  if (!groups) {
    return 0;
  }
  return groups.reduce((total, group) => {
    if ((0, _type_guards.isWaffleMapGroupWithGroups)(group)) {
      return group.groups.reduce((subTotal, subGroup) => {
        if ((0, _type_guards.isWaffleMapGroupWithNodes)(subGroup)) {
          return subTotal > subGroup.nodes.length ? subTotal : subGroup.nodes.length;
        }
        return subTotal;
      }, total);
    }
    if ((0, _type_guards.isWaffleMapGroupWithNodes)(group)) {
      return total > group.nodes.length ? total : group.nodes.length;
    }
    return total;
  }, 0);
}
const getTotalItemsOfGroup = group => getTotalItems([group]);
function applyWaffleMapLayout(groups, width, height) {
  if (groups.length === 0) {
    return [];
  }
  const levels = (0, _type_guards.isWaffleMapGroupWithGroups)((0, _lodash.first)(groups)) ? 2 : 1;
  const totalItems = getTotalItems(groups);
  const squareSize = Math.round((0, _size_of_squares.sizeOfSquares)(width, height, totalItems, levels));
  const largestCount = getLargestCount(groups);
  return (0, _lodash.sortBy)(groups, getTotalItemsOfGroup).reverse().map(group => {
    if ((0, _type_guards.isWaffleMapGroupWithGroups)(group)) {
      const columns = getColumns(largestCount, width, height);
      const groupOfNodes = group.groups;
      const subGroups = (0, _lodash.sortBy)(groupOfNodes, getTotalItemsOfGroup).reverse().filter(_type_guards.isWaffleMapGroupWithNodes).map(subGroup => {
        return {
          ...subGroup,
          count: subGroup.nodes.length,
          columns,
          width: columns * squareSize,
          squareSize
        };
      });
      return {
        ...group,
        groups: subGroups,
        count: getTotalItems([group]),
        squareSize
      };
    }
    if ((0, _type_guards.isWaffleMapGroupWithNodes)(group)) {
      const columns = getColumns(Math.max(group.nodes.length, largestCount), width, height);
      return {
        ...group,
        count: group.nodes.length,
        squareSize,
        width: columns * squareSize
      };
    }
    return group;
  });
}