"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useLogAnalysisModule = void 0;
var _react = require("react");
var _public = require("@kbn/observability-shared-plugin/public");
var _use_log_ml_job_id_formats_shim = require("../../../pages/logs/shared/use_log_ml_job_id_formats_shim");
var _use_kibana = require("../../../hooks/use_kibana");
var _use_tracked_promise = require("../../../hooks/use_tracked_promise");
var _log_analysis_module_status = require("./log_analysis_module_status");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useLogAnalysisModule = ({
  sourceConfiguration,
  idFormat,
  moduleDescriptor
}) => {
  const {
    services
  } = (0, _use_kibana.useKibanaContextForPlugin)();
  const {
    spaceId,
    sourceId: logViewId,
    timestampField,
    runtimeMappings
  } = sourceConfiguration;
  const [moduleStatus, dispatchModuleStatus] = (0, _log_analysis_module_status.useModuleStatus)(moduleDescriptor.jobTypes);
  const {
    migrateIdFormat
  } = (0, _use_log_ml_job_id_formats_shim.useLogMlJobIdFormatsShimContext)();
  const trackMetric = (0, _public.useUiTracker)({
    app: 'infra_logs'
  });
  const [, fetchJobStatus] = (0, _use_tracked_promise.useTrackedPromise)({
    cancelPreviousOn: 'resolution',
    createPromise: async () => {
      dispatchModuleStatus({
        type: 'fetchingJobStatuses'
      });
      return await moduleDescriptor.getJobSummary(spaceId, logViewId, idFormat, services.http.fetch);
    },
    onResolve: jobResponse => {
      dispatchModuleStatus({
        type: 'fetchedJobStatuses',
        payload: jobResponse,
        spaceId,
        logViewId,
        idFormat
      });
    },
    onReject: () => {
      dispatchModuleStatus({
        type: 'failedFetchingJobStatuses'
      });
    }
  }, [spaceId, logViewId, idFormat]);
  const [, setUpModule] = (0, _use_tracked_promise.useTrackedPromise)({
    cancelPreviousOn: 'resolution',
    createPromise: async (selectedIndices, start, end, datasetFilter) => {
      dispatchModuleStatus({
        type: 'startedSetup'
      });
      const setupResult = await moduleDescriptor.setUpModule(start, end, datasetFilter, {
        indices: selectedIndices,
        sourceId: logViewId,
        spaceId,
        timestampField,
        runtimeMappings
      }, services.http.fetch);
      const jobSummaries = await moduleDescriptor.getJobSummary(spaceId, logViewId, 'hashed', services.http.fetch);
      return {
        setupResult,
        jobSummaries
      };
    },
    onResolve: ({
      setupResult: {
        datafeeds,
        jobs
      },
      jobSummaries
    }) => {
      // Track failures
      if ([...datafeeds, ...jobs].reduce((acc, resource) => [...acc, ...Object.keys(resource)], []).some(key => key === 'error')) {
        const reasons = [...datafeeds, ...jobs].filter(resource => resource.error !== undefined).map(resource => {
          var _resource$error$error, _resource$error, _resource$error$error2;
          return (_resource$error$error = (_resource$error = resource.error) === null || _resource$error === void 0 ? void 0 : (_resource$error$error2 = _resource$error.error) === null || _resource$error$error2 === void 0 ? void 0 : _resource$error$error2.reason) !== null && _resource$error$error !== void 0 ? _resource$error$error : '';
        });
        // NOTE: Lack of indices and a missing field mapping have the same error
        if (reasons.filter(reason => reason.includes('because it has no mappings')).length > 0) {
          trackMetric({
            metric: 'logs_ml_setup_error_bad_indices_or_mappings'
          });
        } else {
          trackMetric({
            metric: 'logs_ml_setup_error_unknown_cause'
          });
        }
      }
      dispatchModuleStatus({
        type: 'finishedSetup',
        datafeedSetupResults: datafeeds,
        jobSetupResults: jobs,
        jobSummaries,
        spaceId,
        logViewId,
        idFormat: 'hashed'
      });
      migrateIdFormat(moduleDescriptor.jobTypes[0]);
    },
    onReject: e => {
      var _e$body;
      dispatchModuleStatus({
        type: 'failedSetup'
      });
      if ((e === null || e === void 0 ? void 0 : (_e$body = e.body) === null || _e$body === void 0 ? void 0 : _e$body.statusCode) === 403) {
        trackMetric({
          metric: 'logs_ml_setup_error_lack_of_privileges'
        });
      }
    }
  }, [moduleDescriptor.setUpModule, spaceId, logViewId, timestampField]);
  const [cleanUpModuleRequest, cleanUpModule] = (0, _use_tracked_promise.useTrackedPromise)({
    cancelPreviousOn: 'resolution',
    createPromise: async () => {
      return await moduleDescriptor.cleanUpModule(spaceId, logViewId, idFormat, services.http.fetch);
    },
    onReject: e => {
      throw new Error(`Failed to clean up previous ML job: ${e}`);
    }
  }, [spaceId, logViewId, idFormat]);
  const isCleaningUp = (0, _react.useMemo)(() => cleanUpModuleRequest.state === 'pending', [cleanUpModuleRequest.state]);
  const cleanUpAndSetUpModule = (0, _react.useCallback)((selectedIndices, start, end, datasetFilter) => {
    dispatchModuleStatus({
      type: 'startedSetup'
    });
    cleanUpModule().then(() => {
      setUpModule(selectedIndices, start, end, datasetFilter);
    }).catch(e => {
      dispatchModuleStatus({
        type: 'failedSetup',
        reason: e.toString()
      });
    });
  }, [cleanUpModule, dispatchModuleStatus, setUpModule]);
  const viewResults = (0, _react.useCallback)(() => {
    dispatchModuleStatus({
      type: 'viewedResults'
    });
  }, [dispatchModuleStatus]);
  const jobIds = (0, _react.useMemo)(() => moduleDescriptor.getJobIds(spaceId, logViewId, idFormat), [moduleDescriptor, spaceId, logViewId, idFormat]);
  return {
    cleanUpAndSetUpModule,
    cleanUpModule,
    fetchJobStatus,
    isCleaningUp,
    jobIds,
    jobStatus: moduleStatus.jobStatus,
    jobSummaries: moduleStatus.jobSummaries,
    lastSetupErrorMessages: moduleStatus.lastSetupErrorMessages,
    moduleDescriptor,
    setUpModule,
    setupStatus: moduleStatus.setupStatus,
    sourceConfiguration,
    viewResults
  };
};
exports.useLogAnalysisModule = useLogAnalysisModule;