"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.setChoice = setChoice;
exports.withChatCompleteSpan = withChatCompleteSpan;
var _inferenceCommon = require("@kbn/inference-common");
var _rxjs = require("rxjs");
var _types = require("util/types");
var _with_active_inference_span = require("./with_active_inference_span");
var _types2 = require("./types");
var _flatten_attributes = require("./util/flatten_attributes");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function addEvent(span, event) {
  if (!span.isRecording()) {
    return span;
  }
  const flattened = (0, _flatten_attributes.flattenAttributes)(event.body);
  return span.addEvent(event.name, {
    ...flattened,
    ...event.attributes
  });
}
function setChoice(span, {
  content,
  toolCalls
}) {
  addEvent(span, {
    name: _types2.GenAISemanticConventions.GenAIChoice,
    body: {
      finish_reason: toolCalls.length ? 'tool_calls' : 'stop',
      index: 0,
      message: {
        ...mapAssistantResponse({
          content,
          toolCalls
        })
      }
    }
  });
}
function setTokens(span, {
  prompt,
  completion,
  cached
}) {
  if (!span.isRecording()) {
    return;
  }
  span.setAttributes({
    [_types2.GenAISemanticConventions.GenAIUsageInputTokens]: prompt,
    [_types2.GenAISemanticConventions.GenAIUsageOutputTokens]: completion,
    [_types2.GenAISemanticConventions.GenAIUsageCachedInputTokens]: cached !== null && cached !== void 0 ? cached : 0
  });
}
function getUserMessageEvent(message) {
  return {
    name: _types2.GenAISemanticConventions.GenAIUserMessage,
    body: {
      content: typeof message.content === 'string' ? message.content : JSON.stringify(message.content),
      role: 'user'
    }
  };
}
function getAssistantMessageEvent(message) {
  return {
    name: _types2.GenAISemanticConventions.GenAIAssistantMessage,
    body: mapAssistantResponse({
      content: message.content,
      toolCalls: message.toolCalls
    })
  };
}
function getToolMessageEvent(message) {
  return {
    name: _types2.GenAISemanticConventions.GenAIToolMessage,
    body: {
      role: 'tool',
      id: message.toolCallId,
      content: typeof message.response === 'string' ? message.response : JSON.stringify(message.response)
    }
  };
}
function mapAssistantResponse({
  content,
  toolCalls
}) {
  return {
    content: content || null,
    role: 'assistant',
    tool_calls: toolCalls === null || toolCalls === void 0 ? void 0 : toolCalls.map(toolCall => {
      return {
        function: {
          name: toolCall.function.name,
          arguments: typeof toolCall.function.arguments === 'string' ? toolCall.function.arguments : JSON.stringify(toolCall.function.arguments)
        },
        id: toolCall.toolCallId,
        type: 'function'
      };
    })
  };
}

/**
 * Wrapper around {@link withActiveInferenceSpan} that sets the right attributes for a chat operation span.
 * @param options
 * @param cb
 */

function withChatCompleteSpan(options, cb) {
  var _model$family, _model$provider;
  const {
    system,
    messages,
    model,
    toolChoice,
    tools,
    ...attributes
  } = options;
  const next = (0, _with_active_inference_span.withActiveInferenceSpan)('ChatComplete', {
    attributes: {
      ...attributes,
      [_types2.GenAISemanticConventions.GenAIOperationName]: 'chat',
      [_types2.GenAISemanticConventions.GenAIResponseModel]: (_model$family = model === null || model === void 0 ? void 0 : model.family) !== null && _model$family !== void 0 ? _model$family : 'unknown',
      [_types2.GenAISemanticConventions.GenAISystem]: (_model$provider = model === null || model === void 0 ? void 0 : model.provider) !== null && _model$provider !== void 0 ? _model$provider : 'unknown',
      [_types2.ElasticGenAIAttributes.InferenceSpanKind]: 'LLM',
      [_types2.ElasticGenAIAttributes.Tools]: tools ? JSON.stringify(tools) : undefined,
      [_types2.ElasticGenAIAttributes.ToolChoice]: toolChoice ? JSON.stringify(toolChoice) : toolChoice
    }
  }, span => {
    if (!span) {
      return cb();
    }
    if (system) {
      addEvent(span, {
        name: _types2.GenAISemanticConventions.GenAISystemMessage,
        body: {
          content: system,
          role: 'system'
        }
      });
    }
    messages.map(message => {
      switch (message.role) {
        case _inferenceCommon.MessageRole.User:
          return getUserMessageEvent(message);
        case _inferenceCommon.MessageRole.Assistant:
          return getAssistantMessageEvent(message);
        case _inferenceCommon.MessageRole.Tool:
          return getToolMessageEvent(message);
      }
    }).forEach(event => {
      addEvent(span, event);
    });
    const result = cb(span);
    if ((0, _rxjs.isObservable)(result)) {
      return result.pipe((0, _rxjs.tap)({
        next: value => {
          if ((0, _inferenceCommon.isChatCompletionMessageEvent)(value)) {
            setChoice(span, {
              content: value.content,
              toolCalls: value.toolCalls
            });
          } else if ((0, _inferenceCommon.isChatCompletionTokenCountEvent)(value)) {
            setTokens(span, value.tokens);
          }
        }
      }));
    }
    if ((0, _types.isPromise)(result)) {
      return result.then(value => {
        setChoice(span, {
          content: value.content,
          toolCalls: value.toolCalls
        });
        if (value.tokens) {
          setTokens(span, value.tokens);
        }
        return value;
      });
    }
    return result;
  });
  return next;
}