"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deanonymizeMessage = deanonymizeMessage;
var _inferenceCommon = require("@kbn/inference-common");
var _events = require("@kbn/inference-common/src/chat_complete/events");
var _rxjs = require("rxjs");
var _deanonymize = require("./deanonymize");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function deanonymizeMessage(anonymization) {
  if (!anonymization.anonymizations.length) {
    return _rxjs.identity;
  }
  return source$ => {
    return source$.pipe(
    // Filter out original chunk events (we recreate a single deanonymized chunk later)
    (0, _rxjs.filter)(event => event.type !== _events.ChatCompletionEventType.ChatCompletionChunk),
    // Process message events and create a new chunk plus the message
    (0, _rxjs.mergeMap)(event => {
      if (event.type === _events.ChatCompletionEventType.ChatCompletionMessage) {
        // Create assistant message structure for deanonymization
        const message = {
          content: event.content,
          toolCalls: event.toolCalls,
          role: _inferenceCommon.MessageRole.Assistant
        };
        const {
          message: {
            content: deanonymizedContent,
            toolCalls: deanonymizedToolCalls
          },
          deanonymizations
        } = (0, _deanonymize.deanonymize)(message, anonymization.anonymizations);

        // Create deanonymized input messages metadata
        const deanonymizedInput = anonymization.messages.map(msg => {
          const deanonymization = (0, _deanonymize.deanonymize)(msg, anonymization.anonymizations);
          return {
            message: deanonymization.message,
            deanonymizations: deanonymization.deanonymizations
          };
        });

        // Create deanonymized output metadata
        const deanonymizedOutput = {
          message: {
            content: deanonymizedContent,
            toolCalls: deanonymizedToolCalls,
            role: _inferenceCommon.MessageRole.Assistant
          },
          deanonymizations
        };

        // Create a new chunk with the complete deanonymized content
        const completeChunk = {
          type: _events.ChatCompletionEventType.ChatCompletionChunk,
          content: deanonymizedContent,
          tool_calls: deanonymizedToolCalls.map((tc, idx) => ({
            index: idx,
            toolCallId: tc.toolCallId,
            function: {
              name: tc.function.name,
              arguments: JSON.stringify(tc.function.arguments) || ''
            }
          })),
          deanonymized_input: deanonymizedInput,
          deanonymized_output: deanonymizedOutput
        };

        // Create deanonymized message event
        const deanonymizedMsg = {
          ...event,
          content: deanonymizedContent,
          toolCalls: deanonymizedToolCalls,
          deanonymized_input: deanonymizedInput,
          deanonymized_output: deanonymizedOutput
        };

        // Emit new chunk first, then message
        return (0, _rxjs.of)(completeChunk, deanonymizedMsg);
      }

      // Pass through other events unchanged
      return (0, _rxjs.of)(event);
    }));
  };
}