"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useMappingsStateListener = void 0;
var _react = require("react");
var _lodash = require("lodash");
var _lib = require("./lib");
var _mappings_state_context = require("./mappings_state_context");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useMappingsStateListener = ({
  onChange,
  value,
  status
}) => {
  const state = (0, _mappings_state_context.useMappingsState)();
  const dispatch = (0, _mappings_state_context.useDispatch)();
  const {
    fields: mappedFields,
    runtime: runtimeFields
  } = value !== null && value !== void 0 ? value : {};
  const parsedFieldsDefaultValue = (0, _react.useMemo)(() => (0, _lib.normalize)(mappedFields), [mappedFields]);
  const parsedRuntimeFieldsDefaultValue = (0, _react.useMemo)(() => (0, _lib.normalizeRuntimeFields)(runtimeFields), [runtimeFields]);
  const fieldTypesOptions = (0, _react.useMemo)(() => {
    const allFieldsTypes = (0, _lib.getAllFieldTypesFromState)((0, _lib.deNormalize)((0, _lib.normalize)(mappedFields)));
    return allFieldsTypes.map(dataType => ({
      checked: undefined,
      label: (0, _lib.getTypeLabelFromField)({
        type: dataType
      }),
      'data-test-subj': `indexDetailsMappingsSelectFilter-${dataType}`
    }));
  }, [mappedFields]);
  const calculateStatus = (fieldStatus, rootLevelFields) => {
    if (fieldStatus) return fieldStatus;
    return rootLevelFields.length === 0 ? 'creatingField' : 'idle';
  };
  (0, _react.useEffect)(() => {
    // If we are creating a new field, but haven't entered any name
    // it is valid and we can byPass its form validation (that requires a "name" to be defined)
    const isFieldFormVisible = state.fieldForm !== undefined;
    const emptyNameValue = isFieldFormVisible && (state.fieldForm.data.internal.name === undefined || state.fieldForm.data.internal.name.trim() === '');
    const bypassFieldFormValidation = state.documentFields.status === 'creatingField' && emptyNameValue;
    if (onChange) {
      onChange({
        // Output a mappings object from the user's input.
        getData: () => {
          // Pull the mappings properties from the current editor
          const fields = state.documentFields.editor === 'json' ? state.fieldsJsonEditor.format() : (0, _lib.deNormalize)(state.fields);

          // Get the runtime fields
          const runtime = (0, _lib.deNormalizeRuntimeFields)(state.runtimeFields);
          const configurationData = state.configuration.data.format();
          const templatesData = state.templates.data.format();
          const output = {
            ...(0, _lib.stripUndefinedValues)({
              ...configurationData,
              ...templatesData
            })
          };

          // Mapped fields
          if (fields && Object.keys(fields).length > 0) {
            output.properties = fields;
          }

          // Runtime fields
          if (runtime && Object.keys(runtime).length > 0) {
            output.runtime = runtime;
          }
          return Object.keys(output).length > 0 ? output : undefined;
        },
        validate: async () => {
          const configurationFormValidator = state.configuration.submitForm !== undefined ? new Promise(async (resolve, reject) => {
            try {
              const {
                isValid
              } = await state.configuration.submitForm();
              resolve(isValid);
            } catch (error) {
              reject(error);
            }
          }) : Promise.resolve(true);
          const templatesFormValidator = state.templates.submitForm !== undefined ? new Promise(async (resolve, reject) => {
            try {
              const {
                isValid
              } = await state.templates.submitForm();
              resolve(isValid);
            } catch (error) {
              reject(error);
            }
          }) : Promise.resolve(true);
          const promisesToValidate = [configurationFormValidator, templatesFormValidator];
          if (state.fieldForm !== undefined && !bypassFieldFormValidation) {
            promisesToValidate.push(state.fieldForm.validate());
          }
          return Promise.all(promisesToValidate).then(validationArray => {
            const isValid = validationArray.every(Boolean) && state.fieldsJsonEditor.isValid;
            dispatch({
              type: 'validity:update',
              value: isValid
            });
            return isValid;
          });
        },
        isValid: state.isValid
      });
    }
  }, [state, onChange, dispatch]);
  (0, _react.useEffect)(() => {
    /**
     * If the value has changed that probably means that we have loaded
     * new data from JSON. We need to update our state with the new mappings.
     */
    if (value === undefined) {
      return;
    }
    dispatch({
      type: 'editor.replaceMappings',
      value: {
        configuration: value.configuration,
        templates: value.templates,
        fields: parsedFieldsDefaultValue,
        documentFields: {
          status: calculateStatus(status, parsedFieldsDefaultValue.rootLevelFields),
          editor: 'default'
        },
        runtimeFields: parsedRuntimeFieldsDefaultValue,
        filter: {
          selectedOptions: fieldTypesOptions,
          filteredFields: (0, _lib.getFieldsFromState)(parsedFieldsDefaultValue),
          selectedDataTypes: []
        }
      }
    });
    dispatch({
      type: 'editor.replaceViewMappings',
      value: {
        fields: (0, _lodash.cloneDeep)(parsedFieldsDefaultValue)
      }
    });
  }, [value, parsedFieldsDefaultValue, dispatch, status, parsedRuntimeFieldsDefaultValue, fieldTypesOptions]);
};
exports.useMappingsStateListener = useMappingsStateListener;