"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IndexPatternAdapter = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _create_or_update_index = require("./create_or_update_index");
var _index_adapter = require("./index_adapter");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class IndexPatternAdapter extends _index_adapter.IndexAdapter {
  constructor(prefix, options) {
    super(`${prefix}-*`, options); // make indexTemplate `indexPatterns` match all index names
    (0, _defineProperty2.default)(this, "installationPromises", void 0);
    (0, _defineProperty2.default)(this, "installIndexPromise", void 0);
    this.prefix = prefix;
    this.installationPromises = new Map();
  }

  /** Method to create/update the templates, update existing indices and setup internal state for the adapter. */
  async install(params) {
    this.installIndexPromise = this._install(params);
    await this.installIndexPromise;
  }
  async _install(params) {
    const {
      logger,
      pluginStop$,
      tasksTimeoutMs
    } = params;
    await this.installTemplates(params);
    const esClient = await params.esClient;
    const installFn = this.getInstallFn({
      logger,
      pluginStop$,
      tasksTimeoutMs
    });

    // Update existing specific indices
    await installFn((0, _create_or_update_index.updateIndices)({
      name: this.name,
      // `${prefix}-*`
      esClient,
      logger,
      totalFieldsLimit: this.totalFieldsLimit,
      writeIndexOnly: this.writeIndexOnly
    }), `update specific indices`);

    // Define the function to create concrete indices on demand
    return async name => installFn((0, _create_or_update_index.createIndex)({
      name,
      esClient,
      logger
    }), `create ${name} index`);
  }

  /**
   * Method to create the index for a given index suffix.
   * Stores the installations promises to avoid concurrent installations for the same index.
   * Index creation will only be attempted once per index suffix and existence will be checked before creating.
   */
  async createIndex(indexSuffix) {
    if (!this.installIndexPromise) {
      throw new Error('Cannot installIndex before install');
    }
    const existingInstallation = this.installationPromises.get(indexSuffix);
    if (existingInstallation) {
      return existingInstallation;
    }
    const indexName = this.getIndexName(indexSuffix);

    // Awaits for installIndexPromise to resolve to ensure templates are installed before the specific index is created.
    // This is a safety measure since the initial `install` call may not be awaited from the plugin lifecycle caller.
    // However, the promise will most likely be already fulfilled by the time `createIndex` is called, so this is a no-op.
    const installation = this.installIndexPromise.then(installIndex => installIndex(indexName)).catch(err => {
      this.installationPromises.delete(indexSuffix);
      throw err;
    });
    this.installationPromises.set(indexSuffix, installation);
    return installation;
  }

  /** Method to get the full index name for a given index suffix. */
  getIndexName(indexSuffix) {
    return `${this.prefix}-${indexSuffix}`;
  }

  /** Method to get the full index name for a given index suffix. It returns undefined if the index does not exist. */
  async getInstalledIndexName(indexSuffix) {
    const existingInstallation = this.installationPromises.get(indexSuffix);
    if (!existingInstallation) {
      return undefined;
    }
    return existingInstallation.then(() => this.getIndexName(indexSuffix)).catch(() => undefined);
  }
}
exports.IndexPatternAdapter = IndexPatternAdapter;