"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findConsistentSplitChars = findConsistentSplitChars;
var _lodash = require("lodash");
var _constants = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Analyzes columns of tokens across multiple lines to determine which split characters
 * are consistently used in each column. It returns a list of consistent split characters
 * for each column, including additional quote split characters.
 *
 * @param columnsPerLine - A 2D array where each row represents a line, and each column contains tokens.
 * @returns A 2D array where each entry corresponds to a column and contains the consistent split
 */
function findConsistentSplitChars(columnsPerLine) {
  const splitTokenCountPerColumn = [];
  const consistentTokensPerColumn = [];
  const [splitCharsToCheck, quoteSplitChars] = (0, _lodash.partition)(_constants.TOKEN_SPLIT_CHARS, token => !_constants.ALL_CAPTURE_CHARS.includes(token));
  columnsPerLine.forEach(columns => {
    columns.forEach((column, idx) => {
      const counter = Object.fromEntries(splitCharsToCheck.map(token => [token, 0]));
      column.tokens.forEach(token => {
        if (splitCharsToCheck.includes(token.value)) {
          counter[token.value]++;
        }
      });
      if (!splitTokenCountPerColumn[idx]) {
        consistentTokensPerColumn[idx] = splitCharsToCheck.concat();
        splitTokenCountPerColumn[idx] = counter;
        return;
      }
      consistentTokensPerColumn[idx].forEach(splitToken => {
        const countForThisRow = counter[splitToken];
        const previousCount = splitTokenCountPerColumn[idx][splitToken];
        if (countForThisRow !== previousCount) {
          (0, _lodash.pull)(consistentTokensPerColumn[idx], splitToken);
        }
      });
    });
  });
  return consistentTokensPerColumn.map((tokens, idx) => {
    return splitCharsToCheck.filter(token => {
      return tokens.includes(token);
    }).concat(quoteSplitChars);
  });
}