"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.VERSION = exports.TYPE = exports.AutoInstallContentPackagesTask = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _pMap = _interopRequireDefault(require("p-map"));
var _server = require("@kbn/core/server");
var _task = require("@kbn/task-manager-plugin/server/task");
var _elasticsearch = require("@elastic/elasticsearch");
var _services = require("../services");
var Registry = _interopRequireWildcard(require("../services/epm/registry"));
var _constants = require("../constants");
var _packages = require("../services/epm/packages");
var _get_prerelease_setting = require("../services/epm/packages/get_prerelease_setting");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TYPE = exports.TYPE = 'fleet:auto-install-content-packages-task';
const VERSION = exports.VERSION = '1.0.2';
const TITLE = 'Fleet Auto Install Content Packages Task';
const SCOPE = ['fleet'];
const DEFAULT_INTERVAL = '10m';
const TIMEOUT = '5m';
const CONTENT_PACKAGES_CACHE_TTL = 1000 * 60 * 60; // 1 hour

class AutoInstallContentPackagesTask {
  constructor(setupContract) {
    var _config$taskInterval;
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "wasStarted", false);
    (0, _defineProperty2.default)(this, "taskInterval", void 0);
    (0, _defineProperty2.default)(this, "discoveryMap", void 0);
    (0, _defineProperty2.default)(this, "discoveryMapLastFetched", 0);
    (0, _defineProperty2.default)(this, "lastPrerelease", false);
    (0, _defineProperty2.default)(this, "start", async ({
      taskManager
    }) => {
      if (!taskManager) {
        this.logger.error('[AutoInstallContentPackagesTask] Missing required service during start');
        return;
      }
      this.wasStarted = true;
      this.logger.info(`[AutoInstallContentPackagesTask] Started with interval of [${this.taskInterval}]`);
      try {
        await taskManager.ensureScheduled({
          id: this.taskId,
          taskType: TYPE,
          scope: SCOPE,
          schedule: {
            interval: this.taskInterval
          },
          state: {},
          params: {
            version: VERSION
          }
        });
      } catch (e) {
        this.logger.error(`Error scheduling task AutoInstallContentPackagesTask, error: ${e.message}`, e);
      }
    });
    (0, _defineProperty2.default)(this, "runTask", async (taskInstance, core) => {
      if (!_services.appContextService.getExperimentalFeatures().enableAutoInstallContentPackages) {
        this.logger.debug('[AutoInstallContentPackagesTask] Aborting runTask: auto install content packages feature is disabled');
        return;
      }
      if (!this.wasStarted) {
        this.logger.debug('[AutoInstallContentPackagesTask] runTask Aborted. Task not started yet');
        return;
      }
      // Check that this task is current
      if (taskInstance.id !== this.taskId) {
        this.logger.debug(`[AutoInstallContentPackagesTask] Outdated task version: Got [${taskInstance.id}] from task instance. Current version is [${this.taskId}]`);
        return (0, _task.getDeleteTaskRunResult)();
      }
      this.logger.info(`[runTask()] started`);
      const [coreStart, _startDeps, {
        packageService
      }] = await core.getStartServices();
      const packageClient = packageService.asInternalUser;
      const esClient = coreStart.elasticsearch.client.asInternalUser;
      const soClient = new _server.SavedObjectsClient(coreStart.savedObjects.createInternalRepository());
      const prerelease = await (0, _get_prerelease_setting.getPrereleaseFromSettings)(soClient);
      try {
        if (!this.discoveryMap || this.discoveryMapLastFetched < Date.now() - CONTENT_PACKAGES_CACHE_TTL || this.lastPrerelease !== prerelease) {
          this.lastPrerelease = prerelease;
          this.discoveryMapLastFetched = Date.now();
          this.discoveryMap = await this.getContentPackagesDiscoveryMap(prerelease);
          this.logger.info(`[AutoInstallContentPackagesTask] Fetched content packages discovery map: ${JSON.stringify(this.discoveryMap)}`);
        }
        const installedPackages = await (0, _packages.getInstalledPackages)({
          savedObjectsClient: soClient,
          esClient,
          perPage: _constants.SO_SEARCH_LIMIT,
          sortOrder: 'asc'
        });
        const installedPackagesMap = installedPackages.items.reduce((acc, pkg) => {
          acc[pkg.name] = pkg.version;
          return acc;
        }, {});
        const packagesToInstall = await this.getPackagesToInstall(esClient, installedPackagesMap);
        if (packagesToInstall.length > 0) {
          this.logger.info(`[AutoInstallContentPackagesTask] Content packages to install: ${packagesToInstall.map(pkg => `${pkg.name}@${pkg.version}`).join(', ')}`);
          await this.installPackages(packageClient, packagesToInstall);
        }
        this.endRun('success');
      } catch (err) {
        if (err instanceof _elasticsearch.errors.RequestAbortedError) {
          this.logger.warn(`[AutoInstallContentPackagesTask] request aborted due to timeout: ${err}`);
          this.endRun();
          return;
        }
        this.logger.error(`[AutoInstallContentPackagesTask] error: ${err}`);
        this.endRun('error');
      }
    });
    const {
      core: _core,
      taskManager: _taskManager,
      logFactory,
      config
    } = setupContract;
    this.logger = logFactory.get(this.taskId);
    this.taskInterval = (_config$taskInterval = config.taskInterval) !== null && _config$taskInterval !== void 0 ? _config$taskInterval : DEFAULT_INTERVAL;
    _taskManager.registerTaskDefinitions({
      [TYPE]: {
        title: TITLE,
        timeout: TIMEOUT,
        createTaskRunner: ({
          taskInstance
        }) => {
          return {
            run: async () => {
              return this.runTask(taskInstance, _core);
            },
            cancel: async () => {}
          };
        }
      }
    });
  }
  get taskId() {
    return `${TYPE}:${VERSION}`;
  }
  endRun(msg = '') {
    this.logger.info(`[AutoInstallContentPackagesTask] runTask ended${msg ? ': ' + msg : ''}`);
  }
  async installPackages(packageClient, packagesToInstall) {
    await (0, _pMap.default)(packagesToInstall, async ({
      name,
      version
    }) => {
      try {
        await packageClient.installPackage({
          pkgName: name,
          pkgVersion: version,
          useStreaming: true,
          // Use streaming for content packages
          automaticInstall: true
        });
      } catch (error) {
        this.logger.warn(`[AutoInstallContentPackagesTask] Error installing package ${name}@${version}: ${error.message}`);
      }
    }, {
      concurrency: _constants.MAX_CONCURRENT_EPM_PACKAGES_INSTALLATIONS
    });
  }
  async getPackagesToInstall(esClient, installedPackagesMap) {
    const discoveryMapWithNotInstalledPackages = Object.entries(this.discoveryMap).reduce((acc, [dataset, mapValue]) => {
      const packages = mapValue.packages.filter(pkg => !installedPackagesMap[pkg.name] || installedPackagesMap[pkg.name] !== pkg.version);
      if (packages.length > 0) {
        acc[dataset] = {
          packages
        };
      }
      return acc;
    }, {});
    if (Object.keys(discoveryMapWithNotInstalledPackages).length === 0) {
      return [];
    }
    const datasetsOfInstalledContentPackages = Object.entries(this.discoveryMap).reduce((acc, [dataset, mapValue]) => {
      if (mapValue.packages.every(pkg => installedPackagesMap[pkg.name] && installedPackagesMap[pkg.name] === pkg.version)) {
        acc.push(dataset);
      }
      return acc;
    }, []);
    const datasetsWithData = await this.getDatasetsWithData(esClient, datasetsOfInstalledContentPackages);
    const packagesToInstall = {};
    for (const [dataset, mapValue] of Object.entries(discoveryMapWithNotInstalledPackages)) {
      const packages = mapValue.packages;
      const hasData = datasetsWithData.includes(dataset);
      if (hasData) {
        for (const {
          name,
          version
        } of packages) {
          if (!installedPackagesMap[name] || installedPackagesMap[name] !== version) {
            packagesToInstall[name] = version;
          }
        }
      }
    }
    return Object.entries(packagesToInstall).map(([name, version]) => ({
      name,
      version
    }));
  }
  async getDatasetsWithData(esClient, datasetsOfInstalledContentPackages) {
    const allFleetDataStreams = await _services.dataStreamService.getAllFleetDataStreams(esClient);
    const datasetsWithData = allFleetDataStreams.map(dataStream => dataStream.name.split('-')[1]).filter(dataset => !datasetsOfInstalledContentPackages.includes(dataset));
    this.logger.info(`[AutoInstallContentPackagesTask] Found datasets with data: ${datasetsWithData.join(', ')}`);
    return datasetsWithData;
  }
  async getContentPackagesDiscoveryMap(prerelease) {
    const type = 'content';
    const discoveryMap = {};
    const registryItems = await Registry.fetchList({
      prerelease,
      type
    });
    registryItems.forEach(item => {
      var _item$discovery;
      if ((_item$discovery = item.discovery) !== null && _item$discovery !== void 0 && _item$discovery.datasets) {
        item.discovery.datasets.forEach(field => {
          if (field.name) {
            if (!discoveryMap[field.name]) {
              discoveryMap[field.name] = {
                packages: []
              };
            }
            discoveryMap[field.name].packages.push({
              name: item.name,
              version: item.version
            });
          }
        });
      }
    });
    return discoveryMap;
  }
}
exports.AutoInstallContentPackagesTask = AutoInstallContentPackagesTask;