"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.UninstallTokenService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _crypto = require("crypto");
var _lodash = require("lodash");
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _std = require("@kbn/std");
var _esErrors = require("@kbn/es-errors");
var _common = require("@kbn/spaces-plugin/common");
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
var _utils = require("../../../errors/utils");
var _errors = require("../../../../common/errors");
var _constants = require("../../../constants");
var _app_context = require("../../app_context");
var _agent_policy = require("../../agent_policy");
var _helpers = require("../../spaces/helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getNamespaceFiltering(namespace) {
  if (namespace === _coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING) {
    return `(${_constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE}.attributes.namespaces:default) or (not ${_constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE}.attributes.namespaces:*)`;
  }
  return `${_constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE}.attributes.namespaces:${namespace}`;
}
class UninstallTokenService {
  constructor(esoClient, soClient) {
    (0, _defineProperty2.default)(this, "_soClient", void 0);
    (0, _defineProperty2.default)(this, "isScoped", false);
    (0, _defineProperty2.default)(this, "getUninstallTokenVerificationBatchSize", () => {
      var _config$setup$uninsta, _config$setup;
      /** If `uninstallTokenVerificationBatchSize` is too large, we get an error of `too_many_nested_clauses`.
       *  Assuming that `max_clause_count` >= 1024, and experiencing that batch size should be less than half
       *  than `max_clause_count` with our current query, batch size below 512 should be okay on every env.
       */
      const config = _app_context.appContextService.getConfig();
      return (_config$setup$uninsta = config === null || config === void 0 ? void 0 : (_config$setup = config.setup) === null || _config$setup === void 0 ? void 0 : _config$setup.uninstallTokenVerificationBatchSize) !== null && _config$setup$uninsta !== void 0 ? _config$setup$uninsta : 500;
    });
    (0, _defineProperty2.default)(this, "convertTokenObjectToToken", (policyIdNameDictionary, {
      id: _id,
      attributes,
      created_at: createdAt,
      error
    }) => {
      var _policyIdNameDictiona;
      if (error) {
        throw new _errors.UninstallTokenError(`Error when reading Uninstall Token with id '${_id}'.`);
      }
      this.assertPolicyId(attributes);
      this.assertToken(attributes);
      this.assertCreatedAt(createdAt);
      return {
        id: _id,
        policy_id: attributes.policy_id,
        policy_name: (_policyIdNameDictiona = policyIdNameDictionary[attributes.policy_id]) !== null && _policyIdNameDictiona !== void 0 ? _policyIdNameDictiona : null,
        token: attributes.token || attributes.token_plain,
        created_at: createdAt,
        namespaces: attributes.namespaces
      };
    });
    this.esoClient = esoClient;
    if (soClient) {
      this._soClient = soClient;
      this.isScoped = true;
    }
  }
  getLogger(...childContextPaths) {
    return _app_context.appContextService.getLogger().get('UninstallTokenService', ...childContextPaths);
  }
  scoped(spaceId) {
    return new UninstallTokenService(this.esoClient, _app_context.appContextService.getInternalUserSOClientForSpaceId(spaceId));
  }
  async getToken(id) {
    var _this$soClient$getCur;
    const useSpaceAwareness = this.isScoped && (await (0, _helpers.isSpaceAwarenessEnabled)());
    const namespaceFilter = useSpaceAwareness ? getNamespaceFiltering((_this$soClient$getCur = this.soClient.getCurrentNamespace()) !== null && _this$soClient$getCur !== void 0 ? _this$soClient$getCur : _common.DEFAULT_SPACE_ID) : undefined;
    const filter = `${_constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE}.id: "${_constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE}:${id}"`;
    const tokenObjects = await this.getDecryptedTokenObjects({
      filter: namespaceFilter ? `(${namespaceFilter}) and (${filter})` : filter
    });
    return tokenObjects.length === 1 ? this.convertTokenObjectToToken(await this.getPolicyIdNameDictionary([tokenObjects[0].attributes.policy_id]), tokenObjects[0]) : null;
  }
  prepareSearchString(str, charactersToEscape, wildcard) {
    const strWithoutSpecialCharacters = str === null || str === void 0 ? void 0 : str.replace(new RegExp(charactersToEscape, 'g'), '\\$&').split(/ +/gi).filter(x => x).join(wildcard);
    return strWithoutSpecialCharacters ? wildcard + strWithoutSpecialCharacters + wildcard : undefined;
  }
  prepareExactPolicyIdQuery(policyId) {
    if (!policyId) return undefined;
    // Escape special characters but don't add wildcards for exact matching
    return this.prepareSearchString(policyId, /[@#&*+()\[\]{}|.?~"<]/, '');
  }
  prepareRegexpQuery(str) {
    return this.prepareSearchString(str, /[@#&*+()[\]{}|.?~"<]/, '.*');
  }
  prepareQueryStringQuery(str) {
    return this.prepareSearchString(str, /[":*(){}\\<>]/, '*');
  }
  async searchPoliciesByName(policyNameSearchString) {
    const logger = this.getLogger('searchPoliciesByName');
    logger.debug(`retrieving policies for [${policyNameSearchString}]`);
    const agentPolicySavedObjectType = await (0, _agent_policy.getAgentPolicySavedObjectType)();
    const policyNameFilter = `${agentPolicySavedObjectType}.attributes.name:${policyNameSearchString}`;
    const agentPoliciesSOs = await this.soClient.find({
      type: agentPolicySavedObjectType,
      filter: policyNameFilter
    }).catch(_utils.catchAndSetErrorStackTrace);
    return agentPoliciesSOs.saved_objects.map(attr => attr.id);
  }
  async getTokenMetadata(policyIdSearchTerm, policyNameSearchTerm, page = 1, perPage = 20, excludedPolicyIds) {
    const logger = this.getLogger('getTokenMetadata');
    logger.debug(`getting token metadata with policyIdSearchTerm [${policyIdSearchTerm}] and poliyNameSearchTerm [${policyNameSearchTerm}] and excluded policy ids [${excludedPolicyIds}]`);

    // If theres a search term and not just a policy id, then we use partial, otherwise, use exact matching
    const policyIdFilter = policyNameSearchTerm ? this.prepareRegexpQuery(policyIdSearchTerm) : this.prepareExactPolicyIdQuery(policyIdSearchTerm);
    let policyIdsFoundByName;
    const policyNameSearchString = this.prepareQueryStringQuery(policyNameSearchTerm);
    if (policyNameSearchString) {
      policyIdsFoundByName = await this.searchPoliciesByName(policyNameSearchString);
    }
    let includeFilter;
    if (policyIdFilter || policyIdsFoundByName) {
      includeFilter = [...(policyIdsFoundByName ? policyIdsFoundByName : []), ...(policyIdFilter ? [policyIdFilter] : [])].join('|');
    }
    const tokenObjects = await this.getTokenObjectsByPolicyIdFilter(includeFilter, excludedPolicyIds);
    const tokenObjectsCurrentPage = tokenObjects.slice((page - 1) * perPage, page * perPage);
    const policyIds = tokenObjectsCurrentPage.map(tokenObject => tokenObject._source[_constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE].policy_id);
    const policyIdNameDictionary = await this.getPolicyIdNameDictionary(policyIds);
    const items = tokenObjectsCurrentPage.map(({
      _id,
      _source
    }) => {
      var _policyIdNameDictiona2;
      this.assertPolicyId(_source[_constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE]);
      this.assertCreatedAt(_source.created_at);
      const policyId = _source[_constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE].policy_id;
      return {
        id: _id.replace(`${_constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE}:`, ''),
        policy_id: policyId,
        policy_name: (_policyIdNameDictiona2 = policyIdNameDictionary[policyId]) !== null && _policyIdNameDictiona2 !== void 0 ? _policyIdNameDictiona2 : null,
        created_at: _source.created_at,
        namespaces: _source.namespaces
      };
    });
    return {
      items,
      total: tokenObjects.length,
      page,
      perPage
    };
  }
  async getPolicyIdNameDictionary(policyIds) {
    const logger = this.getLogger('getPolicyIdNameDictionary');
    logger.debug(() => `building policy id name dictionary for [${policyIds.join('|')}]`);
    const agentPolicies = await _agent_policy.agentPolicyService.getByIds(this.soClient, policyIds, {
      ignoreMissing: true
    });
    const warnings = [];
    const response = policyIds.reduce((dict, policyId) => {
      const policy = agentPolicies.find(agentPolicy => agentPolicy.id === policyId);
      if (policy) {
        dict[policyId] = policy.name;
      } else {
        warnings.push(`Policy id [${policyId}] was not found!`);
      }
      return dict;
    }, {});
    if (warnings.length > 0) {
      this.getLogger('getPolicyIdNameDictionary').warn(warnings.join('\n'));
    }
    return response;
  }
  async getDecryptedTokensForPolicyIds(policyIds) {
    const tokenObjects = await this.getDecryptedTokenObjectsForPolicyIds(policyIds);
    const policyIdNameDictionary = await this.getPolicyIdNameDictionary(tokenObjects.map(obj => obj.attributes.policy_id));
    return tokenObjects.map(tokenObject => this.convertTokenObjectToToken(policyIdNameDictionary, tokenObject));
  }
  async getDecryptedTokenObjectsForPolicyIds(policyIds) {
    const tokenObjectHits = await this.getTokenObjectsByPolicyIdFilter(policyIds);
    if (tokenObjectHits.length === 0) {
      return [];
    }
    const filterEntries = tokenObjectHits.map(({
      _id
    }) => `${_constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE}.id: "${_id}"`);
    let tokenObjectChunks = [];
    try {
      tokenObjectChunks = await (0, _std.asyncMap)((0, _lodash.chunk)(filterEntries, this.getUninstallTokenVerificationBatchSize()), async entries => {
        const filter = entries.join(' or ');
        return this.getDecryptedTokenObjects({
          filter
        });
      });
    } catch (error) {
      if ((0, _esErrors.isResponseError)(error) && error.message.includes('too_many_nested_clauses')) {
        // `too_many_nested_clauses` is considered non-fatal
        const errorMessage = 'Failed to validate uninstall tokens: `too_many_nested_clauses` error received. ' + 'Setting/decreasing the value of `xpack.fleet.setup.uninstallTokenVerificationBatchSize` in your kibana.yml should help. ' + `Current value is ${this.getUninstallTokenVerificationBatchSize()}.`;
        _app_context.appContextService.getLogger().warn(`${errorMessage}: '${error}'`);
        throw new _errors.UninstallTokenError(errorMessage);
      } else {
        throw error;
      }
    }
    return tokenObjectChunks.flat();
  }
  async getDecryptedTokenObjects(options) {
    const tokensFinder = await this.esoClient.createPointInTimeFinderDecryptedAsInternalUser({
      type: _constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE,
      perPage: _constants.SO_SEARCH_LIMIT,
      ...options
    });
    let tokenObjects = [];
    for await (const result of tokensFinder.find()) {
      tokenObjects = result.saved_objects;
      break;
    }
    await tokensFinder.close();
    return tokenObjects;
  }
  async getTokenObjectsByPolicyIdFilter(include, exclude) {
    const logger = this.getLogger('getTokenObjectsByPolicyIdFilter');
    logger.debug(`Retrieving tokens objects using Include [${include}] and exclude [${exclude}]`);
    const bucketSize = 10000;
    const useSpaceAwareness = await (0, _helpers.isSpaceAwarenessEnabled)();
    const filter = this.isScoped && useSpaceAwareness ? getNamespaceFiltering(this.soClient.getCurrentNamespace() || _coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING) : undefined;
    const query = {
      type: _constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE,
      perPage: 0,
      filter,
      aggs: {
        by_policy_id: {
          terms: {
            field: `${_constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE}.attributes.policy_id`,
            size: bucketSize,
            include,
            exclude
          },
          aggs: {
            latest: {
              top_hits: {
                size: 1,
                sort: [{
                  created_at: {
                    order: 'desc'
                  }
                }],
                _source: [`${_constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE}.policy_id`, 'created_at']
              }
            }
          }
        }
      }
    };
    logger.debug(() => `Creating SO point in time finder using query:\n${JSON.stringify(query)}`);

    // encrypted saved objects doesn't decrypt aggregation values so we get
    // the ids first from saved objects to use with encrypted saved objects
    const idFinder = this.soClient.createPointInTimeFinder(query);
    let aggResults = [];
    for await (const result of idFinder.find()) {
      var _result$aggregations, _result$aggregations2;
      if (!(result !== null && result !== void 0 && (_result$aggregations = result.aggregations) !== null && _result$aggregations !== void 0 && _result$aggregations.by_policy_id.buckets) || !Array.isArray(result === null || result === void 0 ? void 0 : (_result$aggregations2 = result.aggregations) === null || _result$aggregations2 === void 0 ? void 0 : _result$aggregations2.by_policy_id.buckets)) {
        break;
      }
      aggResults = result.aggregations.by_policy_id.buckets;
      break;
    }
    const getCreatedAt = soBucket => {
      var _soBucket$latest$hits, _soBucket$latest$hits2;
      return new Date((_soBucket$latest$hits = (_soBucket$latest$hits2 = soBucket.latest.hits.hits[0]._source) === null || _soBucket$latest$hits2 === void 0 ? void 0 : _soBucket$latest$hits2.created_at) !== null && _soBucket$latest$hits !== void 0 ? _soBucket$latest$hits : Date.now()).getTime();
    };

    // sorting and paginating buckets is done here instead of ES,
    // because SO query doesn't support `bucket_sort`
    aggResults.sort((a, b) => getCreatedAt(b) - getCreatedAt(a));
    logger.debug(`Returning [${aggResults.length}] aggregated results`);
    return aggResults.map(bucket => bucket.latest.hits.hits[0]);
  }
  async getHashedTokenForPolicyId(policyId) {
    return (await this.getHashedTokensForPolicyIds([policyId]))[policyId];
  }
  async getHashedTokensForPolicyIds(policyIds) {
    const tokens = await this.getDecryptedTokensForPolicyIds(policyIds);
    return tokens.reduce((acc, {
      policy_id: policyId,
      token
    }) => {
      if (policyId && token) {
        acc[policyId] = this.hashToken(token);
      }
      return acc;
    }, {});
  }
  async getAllHashedTokens() {
    const policyIds = await this.getAllPolicyIds();
    return this.getHashedTokensForPolicyIds(policyIds);
  }
  generateTokenForPolicyId(policyId, force = false) {
    return this.generateTokensForPolicyIds([policyId], force);
  }
  async generateTokensForPolicyIds(policyIds, force = false) {
    if (!policyIds.length) {
      return;
    }
    const existingTokens = new Set();
    if (!force) {
      (await this.getTokenObjectsByPolicyIdFilter(policyIds)).forEach(tokenObject => {
        existingTokens.add(tokenObject._source[_constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE].policy_id);
      });
    }
    const missingTokenPolicyIds = force ? policyIds : policyIds.filter(policyId => !existingTokens.has(policyId));
    const newTokensMap = missingTokenPolicyIds.reduce((acc, policyId) => {
      const token = this.generateToken();
      return {
        ...acc,
        [policyId]: token
      };
    }, {});
    await this.persistTokens(missingTokenPolicyIds, newTokensMap);
    if (force) {
      var _config$setup$agentPo, _config$setup2;
      const config = _app_context.appContextService.getConfig();
      const batchSize = (_config$setup$agentPo = config === null || config === void 0 ? void 0 : (_config$setup2 = config.setup) === null || _config$setup2 === void 0 ? void 0 : _config$setup2.agentPolicySchemaUpgradeBatchSize) !== null && _config$setup$agentPo !== void 0 ? _config$setup$agentPo : 100;
      await (0, _std.asyncForEach)((0, _lodash.chunk)(policyIds, batchSize), async policyIdsBatch => await _agent_policy.agentPolicyService.deployPolicies(this.soClient, policyIdsBatch));
    }
  }
  async generateTokensForAllPolicies(force = false) {
    const policyIds = await this.getAllPolicyIds();
    return this.generateTokensForPolicyIds(policyIds, force);
  }
  async encryptTokens() {
    if (!this.isEncryptionAvailable) {
      return;
    }
    const {
      saved_objects: unencryptedTokenObjects
    } = await this.soClient.find({
      type: _constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE,
      filter: `${_constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE}.attributes.token_plain:* AND (NOT ${_constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE}.attributes.token_plain: "")`
    });
    if (!unencryptedTokenObjects.length) {
      return;
    }
    const bulkUpdateObjects = [];
    for (const unencryptedTokenObject of unencryptedTokenObjects) {
      bulkUpdateObjects.push({
        ...unencryptedTokenObject,
        attributes: {
          ...unencryptedTokenObject.attributes,
          token: unencryptedTokenObject.attributes.token_plain,
          token_plain: ''
        }
      });
    }
    await this.soClient.bulkUpdate(bulkUpdateObjects);
  }
  async getAllPolicyIds() {
    const agentPolicyIdsFetcher = await _agent_policy.agentPolicyService.fetchAllAgentPolicyIds(this.soClient, {
      spaceId: '*'
    });
    const policyIds = [];
    for await (const agentPolicyId of agentPolicyIdsFetcher) {
      policyIds.push(...agentPolicyId);
    }
    return policyIds;
  }
  async persistTokens(policyIds, tokensMap) {
    var _config$setup$agentPo2, _config$setup3;
    if (!policyIds.length) {
      return;
    }
    const config = _app_context.appContextService.getConfig();
    const batchSize = (_config$setup$agentPo2 = config === null || config === void 0 ? void 0 : (_config$setup3 = config.setup) === null || _config$setup3 === void 0 ? void 0 : _config$setup3.agentPolicySchemaUpgradeBatchSize) !== null && _config$setup$agentPo2 !== void 0 ? _config$setup$agentPo2 : 100;
    await (0, _std.asyncForEach)((0, _lodash.chunk)(policyIds, batchSize), async policyIdsBatch => {
      const policies = await _agent_policy.agentPolicyService.getByIds(_app_context.appContextService.getInternalUserSOClientWithoutSpaceExtension(), policyIds.map(id => ({
        id,
        spaceId: '*'
      })));
      const policiesSpacesIndexedById = policies.reduce((acc, p) => {
        acc[p.id] = p.space_ids;
        return acc;
      }, {});
      await this.soClient.bulkCreate(policyIdsBatch.map(policyId => ({
        type: _constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE,
        attributes: this.isEncryptionAvailable ? {
          policy_id: policyId,
          token: tokensMap[policyId],
          namespaces: policiesSpacesIndexedById[policyId]
        } : {
          policy_id: policyId,
          token_plain: tokensMap[policyId],
          namespaces: policiesSpacesIndexedById[policyId]
        }
      })));
    });
  }
  generateToken() {
    return (0, _crypto.randomBytes)(16).toString('hex');
  }
  hashToken(token) {
    if (!token) {
      return '';
    }
    const hash = (0, _crypto.createHash)('sha256');
    hash.update(token);
    return hash.digest('base64');
  }
  get soClient() {
    if (this._soClient) {
      return this._soClient;
    }
    const fakeRequest = {
      headers: {},
      getBasePath: () => '',
      path: '/',
      route: {
        settings: {}
      },
      url: {
        href: {}
      },
      raw: {
        req: {
          url: '/'
        }
      }
    };
    this._soClient = _app_context.appContextService.getSavedObjects().getScopedClient(fakeRequest, {
      excludedExtensions: [_coreSavedObjectsServer.SECURITY_EXTENSION_ID, _coreSavedObjectsServer.SPACES_EXTENSION_ID],
      includedHiddenTypes: [_constants.UNINSTALL_TOKENS_SAVED_OBJECT_TYPE]
    });
    return this._soClient;
  }
  async checkTokenValidityForPolicy(policyId) {
    return await this.checkTokenValidityForPolicies([policyId]);
  }
  async checkTokenValidityForAllPolicies() {
    const policyIds = await this.getAllPolicyIds();
    return await this.checkTokenValidityForPolicies(policyIds);
  }
  async checkTokenValidityForPolicies(policyIds) {
    try {
      const tokenObjects = await this.getDecryptedTokenObjectsForPolicyIds(policyIds);
      const numberOfDecryptionErrors = tokenObjects.filter(({
        error
      }) => error).length;
      if (numberOfDecryptionErrors > 0) {
        return {
          error: new _errors.UninstallTokenError(`Failed to decrypt ${numberOfDecryptionErrors} of ${tokenObjects.length} Uninstall Token(s)`)
        };
      }
      const numberOfTokensWithMissingData = tokenObjects.filter(({
        attributes,
        created_at: createdAt
      }) => !createdAt || !attributes.policy_id || !attributes.token && !attributes.token_plain).length;
      if (numberOfTokensWithMissingData > 0) {
        return {
          error: new _errors.UninstallTokenError(`Failed to validate Uninstall Tokens: ${numberOfTokensWithMissingData} of ${tokenObjects.length} tokens are invalid`)
        };
      }
    } catch (error) {
      if (error instanceof _errors.UninstallTokenError) {
        // known errors are considered non-fatal
        return {
          error
        };
      } else {
        const errorMessage = 'Unknown error happened while checking Uninstall Tokens validity';
        _app_context.appContextService.getLogger().error(`${errorMessage}: '${error}'`);
        throw new _errors.UninstallTokenError(errorMessage);
      }
    }
    return null;
  }
  get isEncryptionAvailable() {
    var _appContextService$ge, _appContextService$ge2;
    return (_appContextService$ge = (_appContextService$ge2 = _app_context.appContextService.getEncryptedSavedObjectsSetup()) === null || _appContextService$ge2 === void 0 ? void 0 : _appContextService$ge2.canEncrypt) !== null && _appContextService$ge !== void 0 ? _appContextService$ge : false;
  }
  assertCreatedAt(createdAt) {
    if (!createdAt) {
      throw new _errors.UninstallTokenError('Invalid uninstall token: Saved object is missing creation date.');
    }
  }
  assertToken(attributes) {
    if (!(attributes !== null && attributes !== void 0 && attributes.token) && !(attributes !== null && attributes !== void 0 && attributes.token_plain)) {
      throw new _errors.UninstallTokenError('Invalid uninstall token: Saved object is missing the token attribute.');
    }
  }
  assertPolicyId(attributes) {
    if (!(attributes !== null && attributes !== void 0 && attributes.policy_id)) {
      throw new _errors.UninstallTokenError('Invalid uninstall token: Saved object is missing the policy id attribute.');
    }
  }
}
exports.UninstallTokenService = UninstallTokenService;