"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.deleteSecretsIfNotReferenced = deleteSecretsIfNotReferenced;
exports.diffSecretPaths = diffSecretPaths;
exports.extractAndUpdateSecrets = extractAndUpdateSecrets;
exports.extractAndWriteSecrets = extractAndWriteSecrets;
exports.findPackagePoliciesUsingSecrets = findPackagePoliciesUsingSecrets;
exports.getPolicySecretPaths = getPolicySecretPaths;
var _lodash = require("lodash");
var _policy_template = require("../../../common/services/policy_template");
var _common = require("../../../common");
var _services = require("../../../common/services");
var _app_context = require("../app_context");
var _package_policy = require("../package_policy");
var _common2 = require("./common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Given a new package policy, extracts any secrets, creates them in Elasticsearch,
 * and returns a new package policy with secret references in place of the
 * original secret values, along with an array of secret references for
 * storage on the package policy object itself.
 */
async function extractAndWriteSecrets(opts) {
  const {
    packagePolicy,
    packageInfo,
    esClient
  } = opts;
  const secretPaths = getPolicySecretPaths(packagePolicy, packageInfo);
  const cloudConnectorsSecretReferences = packagePolicy.supports_cloud_connector && packagePolicy.cloud_connector_id ? getCloudConnectorSecretReferences(packagePolicy, secretPaths) : [];
  if (!secretPaths.length) {
    return {
      packagePolicy,
      secretReferences: []
    };
  }
  const secretsToCreate = secretPaths.filter(secretPath => !!secretPath.value.value && !secretPath.value.value.isSecretRef);
  const hasCloudConnectorSecretReferences = packagePolicy.supports_cloud_connector && packagePolicy.cloud_connector_id && cloudConnectorsSecretReferences.length;
  if (hasCloudConnectorSecretReferences) {
    return {
      packagePolicy,
      secretReferences: cloudConnectorsSecretReferences
    };
  }
  const secrets = await (0, _common2.createSecrets)({
    esClient,
    values: secretsToCreate.map(secretPath => secretPath.value.value)
  });
  const policyWithSecretRefs = getPolicyWithSecretReferences(secretsToCreate, secrets, packagePolicy);
  return {
    packagePolicy: policyWithSecretRefs,
    secretReferences: [...secrets.reduce((acc, secret) => {
      if (Array.isArray(secret)) {
        return [...acc, ...secret.map(({
          id
        }) => ({
          id
        }))];
      }
      return [...acc, {
        id: secret.id
      }];
    }, [])]
  };
}

/**
 * Given a package policy update, extracts any secrets, creates them in Elasticsearch,
 * and returns a package policy update with secret references in place of the
 * original secret values, along with an array of secret references for
 * storage on the package policy object itself.
 */
async function extractAndUpdateSecrets(opts) {
  const {
    oldPackagePolicy,
    packagePolicyUpdate,
    packageInfo,
    esClient
  } = opts;
  const oldSecretPaths = getPolicySecretPaths(oldPackagePolicy, packageInfo);
  const updatedSecretPaths = getPolicySecretPaths(packagePolicyUpdate, packageInfo);
  if (!oldSecretPaths.length && !updatedSecretPaths.length) {
    return {
      packagePolicyUpdate,
      secretReferences: [],
      secretsToDelete: []
    };
  }
  const {
    toCreate,
    toDelete,
    noChange
  } = diffSecretPaths(oldSecretPaths, updatedSecretPaths);
  const secretsToCreate = toCreate.filter(secretPath => !!secretPath.value.value);
  const createdSecrets = await (0, _common2.createSecrets)({
    esClient,
    values: secretsToCreate.map(secretPath => secretPath.value.value)
  });
  const policyWithSecretRefs = getPolicyWithSecretReferences(secretsToCreate, createdSecrets, packagePolicyUpdate);
  const secretReferences = [...noChange.reduce((acc, secretPath) => {
    if (secretPath.value.value.ids) {
      return [...acc, ...secretPath.value.value.ids.map(id => ({
        id
      }))];
    }
    return [...acc, {
      id: secretPath.value.value.id
    }];
  }, []), ...createdSecrets.reduce((acc, secret) => {
    if (Array.isArray(secret)) {
      return [...acc, ...secret.map(({
        id
      }) => ({
        id
      }))];
    }
    return [...acc, {
      id: secret.id
    }];
  }, [])];
  const secretsToDelete = [];
  toDelete.forEach(secretPath => {
    var _secretPath$value$val;
    // check if the previous secret is actually a secret refrerence
    // it may be that secrets were not enabled at the time of creation
    // in which case they are just stored as plain text
    if ((_secretPath$value$val = secretPath.value.value) !== null && _secretPath$value$val !== void 0 && _secretPath$value$val.isSecretRef) {
      if (secretPath.value.value.ids) {
        secretPath.value.value.ids.forEach(id => {
          secretsToDelete.push({
            id
          });
        });
      } else {
        secretsToDelete.push({
          id: secretPath.value.value.id
        });
      }
    }
  });
  return {
    packagePolicyUpdate: policyWithSecretRefs,
    secretReferences,
    secretsToDelete
  };
}

/**
 * Given a list of secret ids, checks to see if they are still referenced by any
 * package policies, and if not, deletes them.
 */
async function deleteSecretsIfNotReferenced(opts) {
  const {
    esClient,
    soClient,
    ids
  } = opts;
  const logger = _app_context.appContextService.getLogger();
  const packagePoliciesUsingSecrets = await findPackagePoliciesUsingSecrets({
    soClient,
    ids
  });
  if (packagePoliciesUsingSecrets.length) {
    packagePoliciesUsingSecrets.forEach(({
      id,
      policyIds
    }) => {
      logger.debug(`Not deleting secret with id ${id} is still in use by package policies: ${policyIds.join(', ')}`);
    });
  }
  const secretsToDelete = ids.filter(id => {
    return !packagePoliciesUsingSecrets.some(packagePolicy => packagePolicy.id === id);
  });
  if (!secretsToDelete.length) {
    return;
  }
  try {
    await (0, _common2.deleteSecrets)({
      esClient,
      ids: secretsToDelete
    });
  } catch (e) {
    logger.warn(`Error cleaning up secrets ${ids.join(', ')}: ${e}`);
  }
}

// Utility functions, exported for testing

async function findPackagePoliciesUsingSecrets(opts) {
  const {
    soClient,
    ids
  } = opts;
  const packagePolicies = await _package_policy.packagePolicyService.list(soClient, {
    kuery: `ingest-package-policies.secret_references.id: (${ids.join(' or ')})`,
    perPage: _common.SO_SEARCH_LIMIT,
    page: 1
  });
  if (!packagePolicies.total) {
    return [];
  }

  // create a map of secret_references.id to package policy id
  const packagePoliciesBySecretId = packagePolicies.items.reduce((acc, packagePolicy) => {
    var _packagePolicy$secret;
    packagePolicy === null || packagePolicy === void 0 ? void 0 : (_packagePolicy$secret = packagePolicy.secret_references) === null || _packagePolicy$secret === void 0 ? void 0 : _packagePolicy$secret.forEach(secretReference => {
      if (Array.isArray(secretReference)) {
        secretReference.forEach(({
          id
        }) => {
          if (!acc[id]) {
            acc[id] = [];
          }
          acc[id].push(packagePolicy.id);
        });
      } else {
        if (!acc[secretReference.id]) {
          acc[secretReference.id] = [];
        }
        acc[secretReference.id].push(packagePolicy.id);
      }
    });
    return acc;
  }, {});
  const res = [];
  for (const id of ids) {
    if (packagePoliciesBySecretId[id]) {
      res.push({
        id,
        policyIds: packagePoliciesBySecretId[id]
      });
    }
  }
  return res;
}
function diffSecretPaths(oldPaths, newPaths) {
  const toCreate = [];
  const toDelete = [];
  const noChange = [];
  const newPathsByPath = (0, _lodash.keyBy)(newPaths, x => x.path.join('.'));
  for (const oldPath of oldPaths) {
    if (!newPathsByPath[oldPath.path.join('.')]) {
      toDelete.push(oldPath);
    }
    const newPath = newPathsByPath[oldPath.path.join('.')];
    if (newPath && newPath.value.value) {
      var _newPath$value;
      const newValue = (_newPath$value = newPath.value) === null || _newPath$value === void 0 ? void 0 : _newPath$value.value;
      if (!(newValue !== null && newValue !== void 0 && newValue.isSecretRef)) {
        toCreate.push(newPath);
        toDelete.push(oldPath);
      } else {
        noChange.push(newPath);
      }
      delete newPathsByPath[oldPath.path.join('.')];
    }
  }
  const remainingNewPaths = Object.values(newPathsByPath);
  return {
    toCreate: [...toCreate, ...remainingNewPaths],
    toDelete,
    noChange
  };
}

/**
 * Given a package policy and a package,
 * returns an array of lodash style paths to all secrets and their current values.
 */
function getPolicySecretPaths(packagePolicy, packageInfo) {
  var _packageInfo$policy_t;
  const packageLevelVarPaths = _getPackageLevelSecretPaths(packagePolicy, packageInfo);
  if (!(packageInfo !== null && packageInfo !== void 0 && (_packageInfo$policy_t = packageInfo.policy_templates) !== null && _packageInfo$policy_t !== void 0 && _packageInfo$policy_t.length) || (0, _policy_template.packageHasNoPolicyTemplates)(packageInfo)) {
    return packageLevelVarPaths;
  }
  const inputSecretPaths = _getInputSecretPaths(packagePolicy, packageInfo);
  return [...packageLevelVarPaths, ...inputSecretPaths];
}

// Other utility functions

function isSecretVar(varDef) {
  return varDef.secret === true;
}
function containsSecretVar(vars) {
  return vars === null || vars === void 0 ? void 0 : vars.some(isSecretVar);
}
function _getPackageLevelSecretPaths(packagePolicy, packageInfo) {
  var _packageInfo$vars;
  const packageSecretVars = ((_packageInfo$vars = packageInfo.vars) === null || _packageInfo$vars === void 0 ? void 0 : _packageInfo$vars.filter(isSecretVar)) || [];
  const packageSecretVarsByName = (0, _lodash.keyBy)(packageSecretVars, 'name');
  const packageVars = Object.entries(packagePolicy.vars || {});
  return packageVars.reduce((vars, [name, configEntry], i) => {
    if (packageSecretVarsByName[name]) {
      vars.push({
        value: configEntry,
        path: ['vars', name]
      });
    }
    return vars;
  }, []);
}
function _getInputSecretPaths(packagePolicy, packageInfo) {
  var _packageInfo$policy_t2;
  if (!(packageInfo !== null && packageInfo !== void 0 && (_packageInfo$policy_t2 = packageInfo.policy_templates) !== null && _packageInfo$policy_t2 !== void 0 && _packageInfo$policy_t2.length)) return [];
  const inputSecretVarDefsByPolicyTemplateAndType = _getInputSecretVarDefsByPolicyTemplateAndType(packageInfo);
  const streamSecretVarDefsByDatasetAndInput = _getStreamSecretVarDefsByDatasetAndInput(packageInfo);
  return packagePolicy.inputs.flatMap((input, inputIndex) => {
    if (!input.vars && !input.streams) {
      return [];
    }
    const currentInputVarPaths = [];
    const inputKey = (0, _services.doesPackageHaveIntegrations)(packageInfo) ? `${input.policy_template}-${input.type}` : input.type;
    const inputVars = Object.entries(input.vars || {});
    if (inputVars.length) {
      inputVars.forEach(([name, configEntry]) => {
        var _inputSecretVarDefsBy;
        if ((_inputSecretVarDefsBy = inputSecretVarDefsByPolicyTemplateAndType[inputKey]) !== null && _inputSecretVarDefsBy !== void 0 && _inputSecretVarDefsBy[name]) {
          currentInputVarPaths.push({
            path: ['inputs', inputIndex.toString(), 'vars', name],
            value: configEntry
          });
        }
      });
    }
    if (input.streams.length) {
      input.streams.forEach((stream, streamIndex) => {
        const streamVarDefs = streamSecretVarDefsByDatasetAndInput[`${stream.data_stream.dataset}-${input.type}`];
        if (streamVarDefs && Object.keys(streamVarDefs).length) {
          Object.entries(stream.vars || {}).forEach(([name, configEntry]) => {
            if (streamVarDefs[name]) {
              currentInputVarPaths.push({
                path: ['inputs', inputIndex.toString(), 'streams', streamIndex.toString(), 'vars', name],
                value: configEntry
              });
            }
          });
        }
      });
    }
    return currentInputVarPaths;
  });
}

// a map of all secret vars for each policyTemplate and input type combo
function _getInputSecretVarDefsByPolicyTemplateAndType(packageInfo) {
  var _packageInfo$policy_t3;
  if (!(packageInfo !== null && packageInfo !== void 0 && (_packageInfo$policy_t3 = packageInfo.policy_templates) !== null && _packageInfo$policy_t3 !== void 0 && _packageInfo$policy_t3.length)) return {};
  const hasIntegrations = (0, _services.doesPackageHaveIntegrations)(packageInfo);
  return packageInfo.policy_templates.reduce((varDefs, policyTemplate) => {
    const inputs = (0, _services.getNormalizedInputs)(policyTemplate);
    inputs.forEach(input => {
      var _input$vars;
      const varDefKey = hasIntegrations ? `${policyTemplate.name}-${input.type}` : input.type;
      const secretVars = input === null || input === void 0 ? void 0 : (_input$vars = input.vars) === null || _input$vars === void 0 ? void 0 : _input$vars.filter(isSecretVar);
      if (secretVars !== null && secretVars !== void 0 && secretVars.length) {
        varDefs[varDefKey] = (0, _lodash.keyBy)(secretVars, 'name');
      }
    });
    return varDefs;
  }, {});
}

// a map of all secret vars for each dataset and input combo
function _getStreamSecretVarDefsByDatasetAndInput(packageInfo) {
  const dataStreams = (0, _services.getNormalizedDataStreams)(packageInfo);
  const streamsByDatasetAndInput = dataStreams.reduce((streams, dataStream) => {
    var _dataStream$streams;
    (_dataStream$streams = dataStream.streams) === null || _dataStream$streams === void 0 ? void 0 : _dataStream$streams.forEach(stream => {
      streams[`${dataStream.dataset}-${stream.input}`] = stream;
    });
    return streams;
  }, {});
  return Object.entries(streamsByDatasetAndInput).reduce((varDefs, [path, stream]) => {
    if (stream.vars && containsSecretVar(stream.vars)) {
      const secretVars = stream.vars.filter(isSecretVar);
      varDefs[path] = (0, _lodash.keyBy)(secretVars, 'name');
    }
    return varDefs;
  }, {});
}

/**
 * Given an array of secret paths, existing secrets, and a package policy, generates a
 * new package policy object that includes resolved secret reference values at each
 * provided path.
 */
function getPolicyWithSecretReferences(secretPaths, secrets, packagePolicy) {
  const result = JSON.parse(JSON.stringify(packagePolicy));
  secretPaths.forEach((secretPath, secretPathIndex) => {
    secretPath.path.reduce((acc, val, secretPathComponentIndex) => {
      if (!acc[val]) {
        acc[val] = {};
      }
      const isLast = secretPathComponentIndex === secretPath.path.length - 1;
      if (isLast) {
        acc[val].value = toVarSecretRef(secrets[secretPathIndex]);
      }
      return acc[val];
    }, result);
  });
  return result;
}

// this is how secrets are stored on the package policy
function toVarSecretRef(secret) {
  if (Array.isArray(secret)) {
    return {
      ids: secret.map(({
        id
      }) => id),
      isSecretRef: true
    };
  }
  return {
    id: secret.id,
    isSecretRef: true
  };
}
function getCloudConnectorSecretReferences(packagePolicy, secretPaths) {
  // For cloud connectors, we need to find secret paths that are already secret references
  if (!(packagePolicy !== null && packagePolicy !== void 0 && packagePolicy.supports_cloud_connector) || !packagePolicy.cloud_connector_id) {
    return [];
  }
  return secretPaths.filter(secretPath => {
    var _secretPath$value, _secretPath$value$val2;
    return !!((_secretPath$value = secretPath.value) !== null && _secretPath$value !== void 0 && _secretPath$value.value) && typeof secretPath.value.value === 'object' && ((_secretPath$value$val2 = secretPath.value.value) === null || _secretPath$value$val2 === void 0 ? void 0 : _secretPath$value$val2.id);
  }).map(secretPath => {
    var _secretPath$value$val3;
    return {
      id: (_secretPath$value$val3 = secretPath.value.value) === null || _secretPath$value$val3 === void 0 ? void 0 : _secretPath$value$val3.id
    };
  });
}