"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSecondaryMetricInfo = getSecondaryMetricInfo;
var _i18n = require("@kbn/i18n");
var _utils = require("@kbn/visualizations-plugin/common/utils");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const TREND_UPWARD = '\u{2191}'; // ↑
const TREND_DOWNWARD = '\u{2193}'; // ↓
const TREND_STABLE = '\u{003D}'; // =

const notAvailable = _i18n.i18n.translate('expressionMetricVis.secondaryMetric.notAvailable', {
  defaultMessage: 'N/A'
});
function getEnhancedNumberSignFormatter(trendConfig) {
  if (!(trendConfig !== null && trendConfig !== void 0 && trendConfig.compareToPrimary)) {
    return;
  }
  const paramsOverride = {
    alwaysShowSign: true
  };
  return {
    number: paramsOverride,
    percent: paramsOverride,
    bytes: paramsOverride
  };
}
function getDeltaValue(rawValue, baselineValue) {
  // Return NAN delta for now if either side of the formula is not a number
  if (rawValue == null || baselineValue == null || !Number.isFinite(baselineValue)) {
    return NaN;
  }
  return rawValue - baselineValue;
}
function getBadgeConfiguration(trendConfig, deltaValue) {
  if (Number.isNaN(deltaValue)) {
    return {
      icon: undefined,
      iconLabel: notAvailable,
      color: trendConfig.palette[1]
    };
  }
  if (deltaValue < 0) {
    return {
      icon: trendConfig.showIcon ? TREND_DOWNWARD : undefined,
      iconLabel: _i18n.i18n.translate('expressionMetricVis.secondaryMetric.trend.decrease', {
        defaultMessage: 'downward direction'
      }),
      color: trendConfig.palette[0]
    };
  }
  if (deltaValue > 0) {
    return {
      icon: trendConfig.showIcon ? TREND_UPWARD : undefined,
      iconLabel: _i18n.i18n.translate('expressionMetricVis.secondaryMetric.trend.increase', {
        defaultMessage: 'upward direction'
      }),
      color: trendConfig.palette[2]
    };
  }
  return {
    icon: trendConfig.showIcon ? TREND_STABLE : undefined,
    iconLabel: _i18n.i18n.translate('expressionMetricVis.secondaryMetric.trend.stable', {
      defaultMessage: 'stable'
    }),
    color: trendConfig.palette[1]
  };
}
function getValueToShow(value, deltaValue, formatter, compareToPrimary) {
  if (!compareToPrimary) {
    return String(value);
  }

  // In comparison mode the NaN delta should be converted to N/A
  if (Number.isNaN(deltaValue)) {
    return notAvailable;
  }
  const formattedDelta = formatter ? formatter(deltaValue) : deltaValue;
  return String(formattedDelta);
}
function getTrendDescription(hasIcon, value, direction) {
  if (hasIcon) {
    return _i18n.i18n.translate('expressionMetricVis.secondaryMetric.trend', {
      defaultMessage: 'Value: {value} - Changed to {direction}',
      values: {
        value,
        direction
      }
    });
  }
  return _i18n.i18n.translate('expressionMetricVis.secondaryMetric.trendnoDifferences', {
    defaultMessage: 'Value: {value} - No differences',
    values: {
      value
    }
  });
}
function getStaticColorInfo(formattedValue, label, staticColor) {
  return {
    value: formattedValue,
    label,
    badgeColor: staticColor
  };
}
function getDynamicColorInfo(trendConfig, rawValue, safeFormattedValue, metricFormatter, label) {
  const deltaFactor = trendConfig.compareToPrimary ? -1 : 1;
  const deltaValue = deltaFactor * getDeltaValue(rawValue, trendConfig.baselineValue);
  const {
    icon,
    color: trendColor,
    iconLabel
  } = getBadgeConfiguration(trendConfig, deltaValue);
  const valueToShow = getValueToShow(safeFormattedValue, deltaValue, metricFormatter, trendConfig.compareToPrimary);
  const trendDescription = !trendConfig.showValue ? getTrendDescription(!!icon, valueToShow, iconLabel) : undefined;
  if (trendConfig.showIcon && !trendConfig.showValue && !icon) {
    return {
      value: '',
      label: '',
      badgeColor: '',
      description: trendDescription
    };
  }
  return {
    value: trendConfig.showValue ? valueToShow : '',
    label,
    badgeColor: trendColor,
    description: trendDescription,
    icon: trendConfig.showIcon ? icon : undefined
  };
}

/** Computes the display information for the Secondary Metric */
function getSecondaryMetricInfo({
  row,
  columns,
  secondaryMetric,
  secondaryLabel,
  trendConfig,
  staticColor
}) {
  var _ref;
  const secondaryMetricColumn = (0, _utils.getColumnByAccessor)(secondaryMetric, columns);
  const secondaryMetricFormatter = (0, _helpers.getMetricFormatter)(secondaryMetric, columns, getEnhancedNumberSignFormatter(trendConfig));
  const label = (_ref = secondaryLabel !== null && secondaryLabel !== void 0 ? secondaryLabel : secondaryMetricColumn === null || secondaryMetricColumn === void 0 ? void 0 : secondaryMetricColumn.name) !== null && _ref !== void 0 ? _ref : '';
  const rawValue = secondaryMetricColumn ? row[secondaryMetricColumn.id] : undefined;
  const formattedValue = secondaryMetricFormatter(rawValue);
  const safeFormattedValue = formattedValue !== null && formattedValue !== void 0 ? formattedValue : notAvailable;
  if (staticColor) {
    return getStaticColorInfo(safeFormattedValue, label, staticColor);
  }
  const hasDynamicColor = trendConfig && (typeof rawValue === 'number' || rawValue == null);
  if (hasDynamicColor) {
    return getDynamicColorInfo(trendConfig, rawValue, safeFormattedValue, secondaryMetricFormatter, label);
  }
  return {
    value: formattedValue !== null && formattedValue !== void 0 ? formattedValue : '',
    label
  };
}