"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.walk = exports.Walker = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _utils = require("../visitor/utils");
var _helpers = require("./helpers");
var _Walker;
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */
/**
 * Iterates over all nodes in the AST and calls the appropriate visitor
 * functions.
 */
class Walker {
  constructor(options) {
    (0, _defineProperty2.default)(this, "aborted", false);
    this.options = options;
  }
  abort() {
    this.aborted = true;
  }
  walk(tree, parent = undefined) {
    if (this.aborted) return;
    if (!tree) return;
    if (Array.isArray(tree)) {
      this.walkList(tree, parent);
    } else if (tree.type === 'command') {
      this.walkCommand(tree, parent);
    } else {
      this.walkExpression(tree, parent);
    }
  }
  walkList(list, parent) {
    if (this.aborted) return;
    const {
      options
    } = this;
    const length = list.length;
    if (options.order === 'backward') {
      for (let i = length - 1; i >= 0; i--) {
        this.walk(list[i], parent);
      }
    } else {
      for (let i = 0; i < length; i++) {
        this.walk(list[i], parent);
      }
    }
  }
  walkCommand(node, parent) {
    var _ref, _options$visitCommand;
    if (this.aborted) return;
    const {
      options
    } = this;
    (_ref = (_options$visitCommand = options.visitCommand) !== null && _options$visitCommand !== void 0 ? _options$visitCommand : options.visitAny) === null || _ref === void 0 ? void 0 : _ref(node, parent, this);
    this.walkList(node.args, node);
  }
  walkOption(node, parent) {
    var _ref2, _options$visitCommand2;
    const {
      options
    } = this;
    (_ref2 = (_options$visitCommand2 = options.visitCommandOption) !== null && _options$visitCommand2 !== void 0 ? _options$visitCommand2 : options.visitAny) === null || _ref2 === void 0 ? void 0 : _ref2(node, parent, this);
    this.walkList(node.args, node);
  }
  walkExpression(node, parent = undefined) {
    if (Array.isArray(node)) {
      const list = node;
      this.walkList(list, parent);
    } else {
      const item = node;
      this.walkSingleAstItem(item, parent);
    }
  }
  walkListLiteral(node, parent) {
    var _ref3, _options$visitListLit;
    const {
      options
    } = this;
    (_ref3 = (_options$visitListLit = options.visitListLiteral) !== null && _options$visitListLit !== void 0 ? _options$visitListLit : options.visitAny) === null || _ref3 === void 0 ? void 0 : _ref3(node, parent, this);
    this.walkList(node.values, node);
  }
  walkSource(node, parent) {
    var _ref4, _options$visitSource;
    const {
      options
    } = this;
    (_ref4 = (_options$visitSource = options.visitSource) !== null && _options$visitSource !== void 0 ? _options$visitSource : options.visitAny) === null || _ref4 === void 0 ? void 0 : _ref4(node, parent, this);
    const children = [];
    if (node.prefix) {
      children.push(node.prefix);
    }
    if (node.index) {
      children.push(node.index);
    }
    if (node.selector) {
      children.push(node.selector);
    }
    this.walkList(children, node);
  }
  walkColumn(node, parent) {
    var _ref5, _options$visitColumn;
    const {
      options
    } = this;
    const {
      args
    } = node;
    (_ref5 = (_options$visitColumn = options.visitColumn) !== null && _options$visitColumn !== void 0 ? _options$visitColumn : options.visitAny) === null || _ref5 === void 0 ? void 0 : _ref5(node, parent, this);
    if (args) {
      this.walkList(args, node);
    }
  }
  walkOrder(node, parent) {
    var _ref6, _options$visitOrder;
    const {
      options
    } = this;
    (_ref6 = (_options$visitOrder = options.visitOrder) !== null && _options$visitOrder !== void 0 ? _options$visitOrder : options.visitAny) === null || _ref6 === void 0 ? void 0 : _ref6(node, parent, this);
    this.walkList(node.args, node);
  }
  walkInlineCast(node, parent) {
    var _ref7, _options$visitInlineC;
    const {
      options
    } = this;
    (_ref7 = (_options$visitInlineC = options.visitInlineCast) !== null && _options$visitInlineC !== void 0 ? _options$visitInlineC : options.visitAny) === null || _ref7 === void 0 ? void 0 : _ref7(node, parent, this);
    this.walkExpression(node.value, node);
  }
  walkFunction(node, parent = undefined) {
    var _ref8, _options$visitFunctio;
    const {
      options
    } = this;
    (_ref8 = (_options$visitFunctio = options.visitFunction) !== null && _options$visitFunctio !== void 0 ? _options$visitFunctio : options.visitAny) === null || _ref8 === void 0 ? void 0 : _ref8(node, parent, this);
    if (node.operator) this.walkSingleAstItem(node.operator, node);
    this.walkList(node.args, node);
  }
  walkMap(node, parent) {
    var _ref9, _options$visitMap;
    const {
      options
    } = this;
    (_ref9 = (_options$visitMap = options.visitMap) !== null && _options$visitMap !== void 0 ? _options$visitMap : options.visitAny) === null || _ref9 === void 0 ? void 0 : _ref9(node, parent, this);
    this.walkList(node.entries, node);
  }
  walkMapEntry(node, parent) {
    var _ref10, _options$visitMapEntr;
    const {
      options
    } = this;
    (_ref10 = (_options$visitMapEntr = options.visitMapEntry) !== null && _options$visitMapEntr !== void 0 ? _options$visitMapEntr : options.visitAny) === null || _ref10 === void 0 ? void 0 : _ref10(node, parent, this);
    if (options.order === 'backward') {
      this.walkSingleAstItem((0, _utils.resolveItem)(node.value), node);
      this.walkSingleAstItem((0, _utils.resolveItem)(node.key), node);
    } else {
      this.walkSingleAstItem((0, _utils.resolveItem)(node.key), node);
      this.walkSingleAstItem((0, _utils.resolveItem)(node.value), node);
    }
  }
  walkQuery(node, parent) {
    var _ref11, _options$visitQuery;
    const {
      options
    } = this;
    (_ref11 = (_options$visitQuery = options.visitQuery) !== null && _options$visitQuery !== void 0 ? _options$visitQuery : options.visitAny) === null || _ref11 === void 0 ? void 0 : _ref11(node, parent, this);
    this.walkList(node.commands, node);
  }
  walkSingleAstItem(node, parent) {
    var _options$visitSingleA;
    if (this.aborted) return;
    if (!node) return;
    const {
      options
    } = this;
    (_options$visitSingleA = options.visitSingleAstItem) === null || _options$visitSingleA === void 0 ? void 0 : _options$visitSingleA.call(options, node, parent, this);
    switch (node.type) {
      case 'query':
        {
          this.walkQuery(node, parent);
          break;
        }
      case 'function':
        {
          this.walkFunction(node, parent);
          break;
        }
      case 'map':
        {
          this.walkMap(node, parent);
          break;
        }
      case 'map-entry':
        {
          this.walkMapEntry(node, parent);
          break;
        }
      case 'option':
        {
          this.walkOption(node, parent);
          break;
        }
      case 'source':
        {
          this.walkSource(node, parent);
          break;
        }
      case 'column':
        {
          this.walkColumn(node, parent);
          break;
        }
      case 'order':
        {
          this.walkOrder(node, parent);
          break;
        }
      case 'literal':
        {
          var _ref12, _options$visitLiteral;
          (_ref12 = (_options$visitLiteral = options.visitLiteral) !== null && _options$visitLiteral !== void 0 ? _options$visitLiteral : options.visitAny) === null || _ref12 === void 0 ? void 0 : _ref12(node, parent, this);
          break;
        }
      case 'list':
        {
          this.walkListLiteral(node, parent);
          break;
        }
      case 'inlineCast':
        {
          this.walkInlineCast(node, parent);
          break;
        }
      case 'identifier':
        {
          var _ref13, _options$visitIdentif;
          (_ref13 = (_options$visitIdentif = options.visitIdentifier) !== null && _options$visitIdentif !== void 0 ? _options$visitIdentif : options.visitAny) === null || _ref13 === void 0 ? void 0 : _ref13(node, parent, this);
          break;
        }
      case 'unknown':
        {
          var _ref14, _options$visitUnknown;
          (_ref14 = (_options$visitUnknown = options.visitUnknown) !== null && _options$visitUnknown !== void 0 ? _options$visitUnknown : options.visitAny) === null || _ref14 === void 0 ? void 0 : _ref14(node, parent, this);
          break;
        }
    }
  }
}
exports.Walker = Walker;
_Walker = Walker;
/**
 * Walks the AST and calls the appropriate visitor functions.
 */
(0, _defineProperty2.default)(Walker, "walk", (tree, options) => {
  const walker = new _Walker(options);
  walker.walk(tree);
  return walker;
});
/**
 * Finds and returns the first node that matches the search criteria.
 *
 * @param tree AST node to start the search from.
 * @param predicate A function that returns true if the node matches the search criteria.
 * @returns The first node that matches the search criteria.
 */
(0, _defineProperty2.default)(Walker, "find", (tree, predicate, options) => {
  let found;
  _Walker.walk(tree, {
    ...options,
    visitAny: (node, parent, walker) => {
      if (!found && predicate(node)) {
        found = node;
        walker.abort();
      }
    }
  });
  return found;
});
/**
 * Finds and returns all nodes that match the search criteria.
 *
 * @param tree AST node to start the search from.
 * @param predicate A function that returns true if the node matches the search criteria.
 * @returns All nodes that match the search criteria.
 */
(0, _defineProperty2.default)(Walker, "findAll", (tree, predicate, options) => {
  const list = [];
  _Walker.walk(tree, {
    ...options,
    visitAny: node => {
      if (predicate(node)) {
        list.push(node);
      }
    }
  });
  return list;
});
/**
 * Matches a single node against a template object. Returns the first node
 * that matches the template. The *template* object is a sparse representation
 * of the node structure, where each property corresponds to a node type or
 * property to match against.
 *
 * - An array matches if the node key is in the array.
 * - A RegExp matches if the node key matches the RegExp.
 * - Any other value matches if the node key is triple-equal to the value.
 *
 * For example, match the first `literal`:
 *
 * ```typescript
 * const literal = Walker.match(ast, { type: 'literal' });
 * ```
 *
 * Find the first `literal` with a specific value:
 *
 * ```typescript
 * const number42 = Walker.match(ast, { type: 'literal', value: 42 });
 * ```
 *
 * Find the first literal of type `integer` or `decimal`:
 *
 * ```typescript
 * const number = Walker.match(ast, {
 *   type: 'literal',
 *   literalType: [ 'integer', 'decimal' ],
 * });
 * ```
 *
 * Finally, you can also match any field by regular expression. Find
 * the first `source` AST node, which has "log" in its name:
 *
 * ```typescript
 * const logSource = Walker.match(ast, { type: 'source', name: /.+log.+/ });
 * ```
 *
 * @param tree AST node to match against the template.
 * @param template Template object to match against the node.
 * @returns The first node that matches the template
 */
(0, _defineProperty2.default)(Walker, "match", (tree, template, options) => {
  const predicate = (0, _helpers.templateToPredicate)(template);
  return _Walker.find(tree, predicate, options);
});
/**
 * Matches all nodes against a template object. Returns all nodes that match
 * the template.
 *
 * @param tree AST node to match against the template.
 * @param template Template object to match against the node.
 * @returns All nodes that match the template
 */
(0, _defineProperty2.default)(Walker, "matchAll", (tree, template, options) => {
  const predicate = (0, _helpers.templateToPredicate)(template);
  return _Walker.findAll(tree, predicate, options);
});
/**
 * Replaces a single node in the AST with a new value. Replaces the first
 * node that matches the template with the new value. Replacement happens
 * in-place, so the original AST is modified.
 *
 * For example, replace "?my_param" parameter with an inlined string literal:
 *
 * ```typescript
 * Walker.replace(ast,
 *  { type: 'literal', literalType: 'param', paramType: 'named',
 *      value: 'my_param' },
 *  Builder.expression.literal.string('This is my string'));
 * ```
 *
 * @param tree AST node to search in.
 * @param matcher A function or template object to match against the node.
 * @param newValue The new value to replace the matched node.
 * @returns The updated node, if a match was found and replaced.
 */
(0, _defineProperty2.default)(Walker, "replace", (tree, matcher, newValue) => {
  const node = typeof matcher === 'function' ? _Walker.find(tree, matcher) : _Walker.match(tree, matcher);
  if (!node) return;
  const replacement = typeof newValue === 'function' ? newValue(node) : newValue;
  (0, _helpers.replaceProperties)(node, replacement);
  return node;
});
/**
 * Replaces all nodes in the AST that match the given template with the new
 * value. Works same as {@link Walker.replace}, but replaces all matching nodes.
 *
 * @param tree AST node to search in.
 * @param matcher A function or template object to match against the node.
 * @param newValue The new value to replace the matched nodes.
 * @returns The updated nodes, if any matches were found and replaced.
 */
(0, _defineProperty2.default)(Walker, "replaceAll", (tree, matcher, newValue) => {
  const nodes = typeof matcher === 'function' ? _Walker.findAll(tree, matcher) : _Walker.matchAll(tree, matcher);
  if (nodes.length === 0) return [];
  for (const node of nodes) {
    const replacement = typeof newValue === 'function' ? newValue(node) : newValue;
    (0, _helpers.replaceProperties)(node, replacement);
  }
  return nodes;
});
/**
 * Walks the AST and extracts all command statements.
 *
 * @param tree AST node to extract parameters from.
 */
(0, _defineProperty2.default)(Walker, "commands", (tree, options) => {
  return _Walker.matchAll(tree, {
    type: 'command'
  }, options);
});
/**
 * Walks the AST and extracts all parameter literals.
 *
 * @param tree AST node to extract parameters from.
 */
(0, _defineProperty2.default)(Walker, "params", (tree, options) => {
  return _Walker.matchAll(tree, {
    type: 'literal',
    literalType: 'param'
  }, options);
});
/**
 * Finds the first function that matches the predicate.
 *
 * @param tree AST node from which to search for a function
 * @param predicateOrName Callback to determine if the function is found or
 *     a string with the function name.
 * @returns The first function that matches the predicate
 */
(0, _defineProperty2.default)(Walker, "findFunction", (tree, predicateOrName) => {
  let found;
  const predicate = typeof predicateOrName === 'string' ? node => node.name === predicateOrName : predicateOrName;
  _Walker.walk(tree, {
    visitFunction: (func, parent, walker) => {
      if (!found && predicate(func)) {
        found = func;
        walker.abort();
      }
    }
  });
  return found;
});
/**
 * Searches for at least one occurrence of a function by name.
 *
 * @param tree AST subtree to search in.
 * @param name Function or expression name to search for.
 * @returns True if the function or expression is found in the AST.
 */
(0, _defineProperty2.default)(Walker, "hasFunction", (tree, name) => {
  return !!_Walker.findFunction(tree, name);
});
/**
 * Returns the parent node of the given child node.
 *
 * For example, if the child node is a source node, this method will return
 * the `FROM` command that contains the source:
 *
 * ```typescript
 * const { ast } = EsqlQuery.fromSrc('FROM index');
 * const child = Walker.match(ast, { type: 'source' });
 * const parent = Walker.parent(ast, child); // FROM
 * const grandParent = Walker.parent(ast, parent); // query expression
 * ```
 *
 * @param child The child node for which to find the parent.
 * @returns The parent node of the child, if found.
 */
(0, _defineProperty2.default)(Walker, "parent", (tree, child) => {
  let found;
  _Walker.walk(tree, {
    visitAny: (node, parent, walker) => {
      if (node === child) {
        found = parent;
        walker.abort();
      }
    }
  });
  return found;
});
/**
 * Returns an array of parent nodes for the given child node.
 * This method traverses the AST upwards from the child node
 * and collects all parent nodes until it reaches the root. The
 * most immediate parent is the first element in the array,
 * and the root node is the last element.
 *
 * @param tree AST node to search in.
 * @param child The child node for which to find the parents.
 * @returns An array of parent nodes for the child, if found.
 */
(0, _defineProperty2.default)(Walker, "parents", (tree, child) => {
  const ancestry = [];
  while (true) {
    const parent = _Walker.parent(tree, child);
    if (!parent) break;
    ancestry.push(parent);
    child = parent;
  }
  return ancestry;
});
/**
 * Visits all comment nodes in the AST. Comments are part of the *hidden*
 * channel and normally not part of the AST. To parse the comments, you
 * need to run the parser with `withFormatting` option set to `true`.
 *
 * @param tree AST node to search in.
 * @param callback Callback function that is called for each comment node.
 *     The callback receives the comment node, the node it is attached to,
 *     and the attachment type (top, left, right, rightSingleLine, bottom).
 */
(0, _defineProperty2.default)(Walker, "visitComments", (tree, callback) => {
  _Walker.walk(tree, {
    visitAny: node => {
      const formatting = node.formatting;
      if (!formatting) return;
      if (formatting.top) {
        for (const decoration of formatting.top) {
          if (decoration.type === 'comment') {
            callback(decoration, node, 'top');
          }
        }
      }
      if (formatting.left) {
        for (const decoration of formatting.left) {
          if (decoration.type === 'comment') {
            callback(decoration, node, 'left');
          }
        }
      }
      if (formatting.right) {
        for (const decoration of formatting.right) {
          if (decoration.type === 'comment') {
            callback(decoration, node, 'right');
          }
        }
      }
      if (formatting.rightSingleLine) {
        callback(formatting.rightSingleLine, node, 'rightSingleLine');
      }
      if (formatting.bottom) {
        for (const decoration of formatting.bottom) {
          if (decoration.type === 'comment') {
            callback(decoration, node, 'bottom');
          }
        }
      }
    }
  });
});
/**
 * Visits all nodes in the AST.
 *
 * @param tree AST node to walk.
 * @param visitAny Callback function to call for each node.
 * @param options Additional options for the walker.
 */
(0, _defineProperty2.default)(Walker, "visitAny", (tree, visitAny, options) => {
  _Walker.walk(tree, {
    ...options,
    visitAny
  });
});
const walk = exports.walk = Walker.walk;