"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PARAM_TYPES_THAT_SUPPORT_IMPLICIT_STRING_CASTING = void 0;
exports.argMatchesParamType = argMatchesParamType;
exports.buildPartialMatcher = buildPartialMatcher;
exports.extractValidExpressionRoot = extractValidExpressionRoot;
exports.getBinaryExpressionOperand = getBinaryExpressionOperand;
exports.getExpressionType = getExpressionType;
exports.getMatchingSignatures = getMatchingSignatures;
exports.getRightmostNonVariadicOperator = getRightmostNonVariadicOperator;
exports.isExpressionComplete = isExpressionComplete;
var _lodash = require("lodash");
var _is = require("../../ast/is");
var _utils = require("../../visitor/utils");
var _functions = require("./functions");
var _operators = require("./operators");
var _shared = require("./shared");
var _literals = require("./literals");
var _walker = require("../../walker");
var _ast = require("./ast");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

// #region type detection

/**
 * Determines the type of the expression
 */
function getExpressionType(root, columns) {
  if (!root) {
    return 'unknown';
  }
  if (Array.isArray(root)) {
    if (root.length === 0) {
      return 'unknown';
    }
    return getExpressionType(root[0], columns);
  }
  if ((0, _is.isLiteral)(root)) {
    if (root.literalType === 'param' && _literals.TIME_SYSTEM_PARAMS.includes(root.text)) {
      return 'keyword';
    }
    return root.literalType;
  }

  // from https://github.com/elastic/elasticsearch/blob/122e7288200ee03e9087c98dff6cebbc94e774aa/docs/reference/esql/functions/kibana/inline_cast.json
  if ((0, _is.isInlineCast)(root)) {
    switch (root.castType) {
      case 'int':
        return 'integer';
      case 'bool':
        return 'boolean';
      case 'string':
        return 'keyword';
      case 'datetime':
        return 'date';
      default:
        return root.castType;
    }
  }
  if ((0, _is.isColumn)(root) && columns) {
    const column = (0, _shared.getColumnForASTNode)(root, {
      columns
    });
    const lastArg = (0, _utils.lastItem)(root.args);
    // If the last argument is a param, we return its type (param literal type)
    // This is useful for cases like `where ??field`
    if ((0, _is.isParamLiteral)(lastArg)) {
      return lastArg.literalType;
    }
    if (!column) {
      return 'unknown';
    }
    if ('hasConflict' in column && column.hasConflict) {
      return 'unknown';
    }
    return column.type;
  }
  if (root.type === 'list') {
    return getExpressionType(root.values[0], columns);
  }
  if ((0, _is.isFunctionExpression)(root)) {
    const fnDefinition = (0, _functions.getFunctionDefinition)(root.name);
    if (!fnDefinition) {
      return 'unknown';
    }

    /**
     * Special case for COUNT(*) because
     * the "*" column doesn't match any
     * of COUNT's function definitions
     */
    if (fnDefinition.name === 'count' && root.args[0] && (0, _is.isColumn)(root.args[0]) && root.args[0].name === '*') {
      return 'long';
    }
    if (fnDefinition.name === 'case' && root.args.length) {
      /**
       * The CASE function doesn't fit our system of function definitions
       * and needs special handling. This is imperfect, but it's a start because
       * at least we know that the final argument to case will never be a conditional
       * expression, always a result expression.
       *
       * One problem with this is that if a false case is not provided, the return type
       * will be null, which we aren't detecting. But this is ok because we consider
       * userDefinedColumns and fields to be nullable anyways and account for that during validation.
       */
      return getExpressionType(root.args[root.args.length - 1], columns);
    }
    const argTypes = root.args.map(arg => getExpressionType(arg, columns));
    const literalMask = root.args.map(arg => (0, _is.isLiteral)(arg));
    const matchingSignatures = getMatchingSignatures(fnDefinition.signatures, argTypes, literalMask, false);
    if (matchingSignatures.length > 0 && argTypes.includes('null')) {
      // if one of the arguments is null, the return type is null.
      // this is true for most (all?) functions in ES|QL
      // though it is not reflected in our function definitions
      // so we handle it here
      return 'null';
    }
    const returnTypes = (0, _lodash.uniq)(matchingSignatures.map(sig => sig.returnType));

    // no signature matched the provided arguments
    if (returnTypes.length === 0) return 'unknown';

    // ambiguous return type (i.e. we can't always identify the true
    // matching signature because we don't always know the types of the parameters)
    if (returnTypes.length > 1) return 'unknown';
    if (returnTypes[0] === 'any') {
      return 'unknown';
    }
    return returnTypes[0];
  }
  return 'unknown';
}

// #endregion type detection

// #region signature matching

// ES implicitly casts string literal arguments to these types when passed to functions that expect these types
// e.g. EXPECTS_DATE('2020-01-01') is valid because the string literal is implicitly cast to a date
const PARAM_TYPES_THAT_SUPPORT_IMPLICIT_STRING_CASTING = exports.PARAM_TYPES_THAT_SUPPORT_IMPLICIT_STRING_CASTING = ['date', 'date_nanos', 'date_period', 'time_duration', 'version', 'ip', 'boolean'];

/**
 * Returns all signatures matching the given types and arity
 * @param definition
 * @param types
 */
function getMatchingSignatures(signatures, givenTypes,
// a boolean array indicating which args are literals
literalMask, acceptUnknown) {
  return signatures.filter(sig => {
    if (!matchesArity(sig, givenTypes.length)) {
      return false;
    }
    return givenTypes.every((givenType, index) => {
      // safe to assume the param is there, because we checked the length above
      const expectedType = unwrapArrayOneLevel(getParamAtPosition(sig, index).type);
      return argMatchesParamType(givenType, expectedType, literalMask[index], acceptUnknown);
    });
  });
}

/**
 * Checks if the given type matches the expected parameter type
 *
 * @param givenType
 * @param expectedType
 * @param givenIsLiteral
 */
function argMatchesParamType(givenType, expectedType, givenIsLiteral, acceptUnknown) {
  if (givenType === expectedType || expectedType === 'any' || givenType === 'param' ||
  // all ES|QL functions accept null, but this is not reflected
  // in our function definitions so we let it through here
  givenType === 'null' ||
  // Check array types
  givenType === unwrapArrayOneLevel(expectedType) ||
  // all functions accept keywords for text parameters
  bothStringTypes(givenType, expectedType)) {
    return true;
  }
  if (givenType === 'unknown') return acceptUnknown;
  if (givenIsLiteral && givenType === 'keyword' && PARAM_TYPES_THAT_SUPPORT_IMPLICIT_STRING_CASTING.includes(expectedType)) return true;
  return false;
}

/**
 * Checks if both types are string types.
 *
 * Functions in ES|QL accept `text` and `keyword` types interchangeably.
 * @param type1
 * @param type2
 * @returns
 */
function bothStringTypes(type1, type2) {
  return (type1 === 'text' || type1 === 'keyword') && (type2 === 'text' || type2 === 'keyword');
}

/**
 * Given an array type for example `string[]` it will return `string`
 */
function unwrapArrayOneLevel(type) {
  return (0, _operators.isArrayType)(type) ? type.slice(0, -2) : type;
}
function matchesArity(signature, arity) {
  if (signature.minParams) {
    return arity >= signature.minParams;
  }
  return arity >= signature.params.filter(({
    optional
  }) => !optional).length && arity <= signature.params.length;
}

/**
 * Given a function signature, returns the parameter at the given position.
 *
 * Takes into account variadic functions (minParams), returning the last
 * parameter if the position is greater than the number of parameters.
 *
 * @param signature
 * @param position
 * @returns
 */
function getParamAtPosition({
  params,
  minParams
}, position) {
  return params.length > position ? params[position] : minParams ? params[params.length - 1] : null;
}

// #endregion signature matching

// #region expression completeness

/**
 * Builds a regex that matches partial strings starting
 * from the beginning of the string.
 *
 * Example:
 * "is null" -> /^i(?:s(?:\s+(?:n(?:u(?:l(?:l)?)?)?)?)?)?$/i
 */
function buildPartialMatcher(str) {
  // Split the string into characters
  const chars = str.split('');

  // Initialize the regex pattern
  let pattern = '';

  // Iterate through the characters and build the pattern
  chars.forEach((char, index) => {
    if (char === ' ') {
      pattern += '\\s+';
    } else {
      pattern += char;
    }
    if (index < chars.length - 1) {
      pattern += '(?:';
    }
  });

  // Close the non-capturing groups
  for (let i = 0; i < chars.length - 1; i++) {
    pattern += ')?';
  }

  // Return the final regex pattern
  return pattern;
}

// Handles: "IS ", "IS N", "IS NU", "IS NUL" with flexible whitespace
const isNullMatcher = new RegExp('is\\s*(' + buildPartialMatcher('nul') + ')?$', 'i');
const isNotNullMatcher = new RegExp('is\\s*(' + buildPartialMatcher('not nul') + ')?$', 'i');

// --- Expression types helpers ---

/**
 * Checks whether an expression is truly complete.
 *
 * (Encapsulates handling of the "is null" and "is not null"
 * checks)
 *
 * @todo use the simpler "getExpressionType(root) !== 'unknown'"
 * as soon as https://github.com/elastic/kibana/issues/199401 is resolved
 */
function isExpressionComplete(expressionType, innerText) {
  return expressionType !== 'unknown' &&
  // see https://github.com/elastic/kibana/issues/199401
  // for the reason we need this string check.
  !(isNullMatcher.test(innerText) || isNotNullMatcher.test(innerText));
}

// #endregion expression completeness

/**
 * Returns the left or right operand of a binary expression function.
 */
function getBinaryExpressionOperand(binaryExpression, side) {
  const left = binaryExpression.args[0];
  const right = binaryExpression.args[1];
  return side === 'left' ? left : right;
}

/**
 * Extracts a valid expression root from an assignment RHS, handling arrays and marker nodes.
 */
function extractValidExpressionRoot(assignmentRhs) {
  let root;
  if (Array.isArray(assignmentRhs)) {
    root = assignmentRhs[0] || undefined;
  } else {
    root = assignmentRhs;
  }
  if (!root || (0, _ast.isMarkerNode)(root)) {
    return undefined;
  }
  return getRightmostNonVariadicOperator(root);
}

/**
 * Finds the rightmost non-variadic operator in an expression tree.
 * Useful for locating the most specific node near the cursor.
 */
function getRightmostNonVariadicOperator(root) {
  if ((root === null || root === void 0 ? void 0 : root.type) !== 'function') {
    return root;
  }
  let rightmostFn = root;
  const walker = new _walker.Walker({
    visitFunction: fn => {
      if (fn.subtype !== 'variadic-call' && fn.location.min > rightmostFn.location.min) {
        rightmostFn = fn;
      }
    }
  });
  walker.walkFunction(root);
  return rightmostFn;
}