"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getFunctionArgsSuggestions = getFunctionArgsSuggestions;
exports.getFunctionsToIgnoreForStats = getFunctionsToIgnoreForStats;
exports.getInsideFunctionsSuggestions = void 0;
exports.isAggFunctionUsedAlready = isAggFunctionUsedAlready;
exports.isTimeseriesAggUsedAlready = isTimeseriesAggUsedAlready;
var _lodash = require("lodash");
var _location = require("../../../commands_registry/location");
var _is = require("../../../ast/is");
var _complete_items = require("../../../commands_registry/complete_items");
var _parser = require("../../../parser");
var _walker = require("../../../walker");
var _all_operators = require("../../all_operators");
var _constants = require("../../constants");
var _types = require("../../types");
var _ast = require("../ast");
var _columns = require("../columns");
var _expressions = require("../expressions");
var _functions = require("../functions");
var _literals = require("../literals");
var _operators = require("../operators");
var _values = require("../values");
var _helpers = require("./helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function checkContentPerDefinition(fn, def) {
  const fnDef = (0, _functions.getFunctionDefinition)(fn.name);
  return !!fnDef && fnDef.type === def || extractFunctionArgs(fn.args).some(arg => checkContentPerDefinition(arg, def));
}
function isAggFunctionUsedAlready(command, argIndex) {
  if (argIndex < 0) {
    return false;
  }
  const arg = command.args[argIndex];
  return (0, _is.isFunctionExpression)(arg) ? checkContentPerDefinition(arg, _types.FunctionDefinitionTypes.AGG) : false;
}
function isTimeseriesAggUsedAlready(command, argIndex) {
  if (argIndex < 0) {
    return false;
  }
  const arg = command.args[argIndex];
  return (0, _is.isFunctionExpression)(arg) ? checkContentPerDefinition(arg, _types.FunctionDefinitionTypes.TIME_SERIES_AGG) : false;
}
function extractFunctionArgs(args) {
  return args.flatMap(arg => (0, _is.isAssignment)(arg) ? arg.args[1] : arg).filter(_is.isFunctionExpression);
}
function getFnContent(fn) {
  return [fn.name].concat(extractFunctionArgs(fn.args).flatMap(getFnContent));
}
function getFunctionsToIgnoreForStats(command, argIndex) {
  if (argIndex < 0) {
    return [];
  }
  const arg = command.args[argIndex];
  return (0, _is.isFunctionExpression)(arg) ? getFnContent(arg) : [];
}
const addCommaIf = (condition, text) => condition ? `${text},` : text;
const getCommandAndOptionWithinFORK = command => {
  var _subCommand;
  let option;
  let subCommand;
  _walker.Walker.walk(command, {
    visitCommandOption: _node => {
      option = _node;
    },
    visitCommand: _node => {
      subCommand = _node;
    }
  });
  return {
    option,
    command: (_subCommand = subCommand) !== null && _subCommand !== void 0 ? _subCommand : command
  };
};
async function getFunctionArgsSuggestions(innerText, commands, getFieldsByType, fullText, offset, context, hasMinimumLicenseRequired) {
  var _astContext$command;
  const astContext = (0, _ast.findAstPosition)(commands, offset);
  const node = astContext.node;
  // If the node is not
  if (!node) {
    return [];
  }
  let command = astContext.command;
  if (((_astContext$command = astContext.command) === null || _astContext$command === void 0 ? void 0 : _astContext$command.name) === 'fork') {
    var _astContext$command2;
    const {
      command: forkCommand
    } = ((_astContext$command2 = astContext.command) === null || _astContext$command2 === void 0 ? void 0 : _astContext$command2.name) === 'fork' ? getCommandAndOptionWithinFORK(astContext.command) : {
      command: undefined
    };
    command = forkCommand || astContext.command;
  }
  const functionNode = node;
  const fnDefinition = (0, _functions.getFunctionDefinition)(functionNode.name);
  // early exit on no hit
  if (!fnDefinition) {
    return [];
  }
  const filteredFnDefinition = {
    ...fnDefinition,
    signatures: (0, _functions.filterFunctionSignatures)(fnDefinition.signatures, hasMinimumLicenseRequired)
  };
  const columnMap = (context === null || context === void 0 ? void 0 : context.columns) || new Map();
  const references = {
    columns: columnMap
  };
  const {
    typesToSuggestNext,
    hasMoreMandatoryArgs,
    enrichedArgs,
    argIndex
  } = (0, _helpers.getValidSignaturesAndTypesToSuggestNext)(functionNode, references, filteredFnDefinition, fullText, offset);
  const arg = enrichedArgs[argIndex];

  // Whether to prepend comma to suggestion string
  // E.g. if true, "fieldName" -> "fieldName, "
  const isCursorFollowedByComma = fullText ? fullText.slice(offset, fullText.length).trimStart().startsWith(',') : false;
  const canBeBooleanCondition =
  // For `CASE()`, there can be multiple conditions, so keep suggesting fields and functions if possible
  fnDefinition.name === 'case' ||
  // If the type is explicitly a boolean condition
  typesToSuggestNext.some(t => t && t.type === 'boolean' && t.name === 'condition');
  const shouldAddComma = hasMoreMandatoryArgs && fnDefinition.type !== _types.FunctionDefinitionTypes.OPERATOR && !isCursorFollowedByComma && !canBeBooleanCondition;
  const shouldAdvanceCursor = hasMoreMandatoryArgs && fnDefinition.type !== _types.FunctionDefinitionTypes.OPERATOR && !isCursorFollowedByComma;
  const suggestedConstants = (0, _lodash.uniq)(typesToSuggestNext.map(d => d.suggestedValues).filter(d => d).flat());
  if (suggestedConstants.length) {
    return (0, _values.buildValueDefinitions)(suggestedConstants, {
      addComma: shouldAddComma,
      advanceCursorAndOpenSuggestions: hasMoreMandatoryArgs
    });
  }
  const suggestions = [];
  const noArgDefined = !arg;
  const isUnknownColumn = arg && (0, _is.isColumn)(arg) && !(0, _columns.getColumnExists)(arg, {
    columns: columnMap
  });
  if (noArgDefined || isUnknownColumn) {
    // ... | EVAL fn( <suggest>)
    // ... | EVAL fn( field, <suggest>)

    const commandArgIndex = command.args.findIndex(cmdArg => !Array.isArray(cmdArg) && cmdArg.location.max >= node.location.max);
    const finalCommandArgIndex = command.name !== 'stats' && command.name !== 'inline stats' ? -1 : commandArgIndex < 0 ? Math.max(command.args.length - 1, 0) : commandArgIndex;
    const finalCommandArg = command.args[finalCommandArgIndex];
    const fnToIgnore = [];
    fnToIgnore.push(...(0, _functions.getAllFunctions)({
      type: _types.FunctionDefinitionTypes.GROUPING
    }).map(({
      name
    }) => name));
    if (command.name !== 'stats' && command.name !== 'inline stats' || (0, _is.isOptionNode)(finalCommandArg) && finalCommandArg.name === 'by') {
      // ignore the current function
      fnToIgnore.push(node.name);
    } else {
      fnToIgnore.push(...getFunctionsToIgnoreForStats(command, finalCommandArgIndex), ...(isAggFunctionUsedAlready(command, finalCommandArgIndex) ? (0, _functions.getAllFunctions)({
        type: _types.FunctionDefinitionTypes.AGG
      }).map(({
        name
      }) => name) : []), ...(isTimeseriesAggUsedAlready(command, finalCommandArgIndex) ? (0, _functions.getAllFunctions)({
        type: _types.FunctionDefinitionTypes.TIME_SERIES_AGG
      }).map(({
        name
      }) => name) : []));
    }
    // Separate the param definitions into two groups:
    // fields should only be suggested if the param isn't constant-only,
    // and constant suggestions should only be given if it is.
    //
    // TODO - consider incorporating the literalOptions into this
    //
    // TODO — improve this to inherit the constant flag from the outer function
    // (e.g. if func1's first parameter is constant-only, any nested functions should
    // inherit that constraint: func1(func2(shouldBeConstantOnly)))
    //
    const constantOnlyParamDefs = typesToSuggestNext.filter(p => p.constantOnly || /_duration/.test(p.type));
    const getTypesFromParamDefs = paramDefs => {
      return Array.from(new Set(paramDefs.map(({
        type
      }) => type)));
    };
    const supportsControls = Boolean(context === null || context === void 0 ? void 0 : context.supportsControls);
    const variables = context === null || context === void 0 ? void 0 : context.variables;

    // Literals
    suggestions.push(...(0, _literals.getCompatibleLiterals)(getTypesFromParamDefs(constantOnlyParamDefs), {
      addComma: shouldAddComma,
      advanceCursorAndOpenSuggestions: hasMoreMandatoryArgs,
      supportsControls
    }, variables));
    const ensureKeywordAndText = types => {
      if (types.includes('keyword') && !types.includes('text')) {
        types.push('text');
      }
      if (types.includes('text') && !types.includes('keyword')) {
        types.push('keyword');
      }
      return types;
    };

    // Fields

    // In most cases, just suggest fields that match the parameter types.
    // But in the case of boolean conditions, we want to suggest fields of any type,
    // since they may be used in comparisons.

    // and we always add a comma at the end if there are more mandatory args
    // but this needs to be refined when full expressions begin to be supported

    suggestions.push(...(0, _helpers.pushItUpInTheList)(await getFieldsByType(
    // For example, in case() where we are expecting a boolean condition
    // we can accept any field types (field1 !== field2)
    canBeBooleanCondition ? ['any'] :
    // @TODO: have a way to better suggest constant only params
    ensureKeywordAndText(getTypesFromParamDefs(typesToSuggestNext.filter(d => !d.constantOnly))), [], {
      addComma: shouldAddComma,
      advanceCursor: shouldAdvanceCursor,
      openSuggestions: shouldAdvanceCursor
    }), true));

    // Functions
    if (typesToSuggestNext.every(d => !d.fieldsOnly)) {
      const location = (0, _location.getLocationInfo)(offset, command, commands, isAggFunctionUsedAlready(command, finalCommandArgIndex)).id;
      suggestions.push(...(0, _functions.getFunctionSuggestions)({
        location,
        returnTypes: canBeBooleanCondition ? ['any'] : ensureKeywordAndText(getTypesFromParamDefs(typesToSuggestNext)),
        ignored: fnToIgnore
      }, hasMinimumLicenseRequired, context === null || context === void 0 ? void 0 : context.activeProduct).map(suggestion => ({
        ...suggestion,
        text: addCommaIf(shouldAddComma, suggestion.text)
      })));
    }
    if (getTypesFromParamDefs(typesToSuggestNext).includes('date') && ['where', 'eval'].includes(command.name) && !_constants.FULL_TEXT_SEARCH_FUNCTIONS.includes(fnDefinition.name) || ['stats', 'inline stats'].includes(command.name) && typesToSuggestNext.some(t => t && t.type === 'date' && t.constantOnly === true)) suggestions.push(...(0, _literals.getDateLiterals)({
      addComma: shouldAddComma,
      advanceCursorAndOpenSuggestions: hasMoreMandatoryArgs
    }));
  }

  // for eval and row commands try also to complete numeric literals with time intervals where possible
  if (arg) {
    if (command.name !== 'stats' && command.name !== 'inline stats') {
      if ((0, _is.isLiteral)(arg) && (0, _types.isNumericType)(arg.literalType)) {
        // ... | EVAL fn(2 <suggest>)
        suggestions.push(...(0, _literals.getCompatibleLiterals)(['time_literal_unit'], {
          addComma: shouldAddComma,
          advanceCursorAndOpenSuggestions: hasMoreMandatoryArgs
        }));
      }
    }
    // Suggest comparison functions for boolean conditions
    if (canBeBooleanCondition) {
      suggestions.push(..._all_operators.comparisonFunctions.map(({
        name,
        description
      }) => ({
        label: name,
        text: name,
        kind: 'Function',
        detail: description
      })));
    }
    if (hasMoreMandatoryArgs) {
      // Suggest a comma if there's another argument for the function
      suggestions.push(_complete_items.commaCompleteItem);
    }
  }

  // For special case of COUNT, suggest * if cursor is in empty spot
  // e.g. count( / ) -> suggest `*`
  if (fnDefinition.name === 'count' && !arg) {
    suggestions.push(_complete_items.allStarConstant);
  }
  return suggestions;
}
function isOperator(node) {
  var _getFunctionDefinitio;
  return ((_getFunctionDefinitio = (0, _functions.getFunctionDefinition)(node.name)) === null || _getFunctionDefinitio === void 0 ? void 0 : _getFunctionDefinitio.type) === _types.FunctionDefinitionTypes.OPERATOR;
}
async function getListArgsSuggestions(innerText, commands, getFieldsByType, columnMap, offset, hasMinimumLicenseRequired, activeProduct) {
  const suggestions = [];
  const {
    command,
    node
  } = (0, _ast.findAstPosition)(commands, offset);

  // node is supposed to be the function who support a list argument (like the "in" operator)
  // so extract the type of the first argument and suggest fields of that type
  if (node && (0, _is.isFunctionExpression)(node)) {
    const list = node === null || node === void 0 ? void 0 : node.args[1];
    if ((0, _is.isList)(list)) {
      const noParens = list.location.min === 0 && list.location.max === 0;
      if (noParens) {
        suggestions.push(_complete_items.listCompleteItem);
        return suggestions;
      }
    }
    const [firstArg] = node.args;
    if ((0, _is.isColumn)(firstArg)) {
      const argType = (0, _expressions.getExpressionType)(firstArg, columnMap);
      if (argType) {
        // do not propose existing columns again
        const otherArgs = (0, _is.isList)(list) ? list.values : node.args.filter(Array.isArray).flat().filter(_is.isColumn);
        suggestions.push(...(await (0, _helpers.getFieldsOrFunctionsSuggestions)([argType], (0, _location.getLocationInfo)(offset, command, commands, false).id, getFieldsByType, {
          functions: true,
          columns: true
        }, {
          ignoreColumns: [firstArg.name, ...otherArgs.map(({
            name
          }) => name)]
        }, hasMinimumLicenseRequired, activeProduct)));
      }
    }
  }
  return suggestions;
}
function isNotEnrichClauseAssigment(node, command) {
  return node.name !== '=' && command.name !== 'enrich';
}

// TODO: merge this into suggestForExpression
const getInsideFunctionsSuggestions = async (query, cursorPosition, callbacks, context) => {
  const innerText = query.substring(0, cursorPosition);
  const correctedQuery = (0, _ast.correctQuerySyntax)(innerText);
  const {
    ast
  } = (0, _parser.parse)(correctedQuery, {
    withFormatting: true
  });
  const {
    node,
    command,
    containingFunction
  } = (0, _ast.findAstPosition)(ast, cursorPosition !== null && cursorPosition !== void 0 ? cursorPosition : 0);
  if (!node) {
    return undefined;
  }
  if (node.type === 'literal' && node.literalType === 'keyword') {
    // command ... "<here>"
    return [];
  }
  if (node.type === 'function') {
    // For now, we don't suggest for expressions within any function besides CASE
    // e.g. CASE(field != /)
    //
    // So, it is handled as a special branch...
    if ((containingFunction === null || containingFunction === void 0 ? void 0 : containingFunction.name) === 'case' && !Array.isArray(node) && (node === null || node === void 0 ? void 0 : node.subtype) === 'binary-expression') {
      var _callbacks$getByType;
      return await (0, _operators.getSuggestionsToRightOfOperatorExpression)({
        queryText: innerText,
        location: (0, _location.getLocationInfo)(cursorPosition !== null && cursorPosition !== void 0 ? cursorPosition : 0, command, ast, false).id,
        rootOperator: node,
        getExpressionType: expression => (0, _expressions.getExpressionType)(expression, context === null || context === void 0 ? void 0 : context.columns),
        getColumnsByType: (_callbacks$getByType = callbacks === null || callbacks === void 0 ? void 0 : callbacks.getByType) !== null && _callbacks$getByType !== void 0 ? _callbacks$getByType : () => Promise.resolve([]),
        hasMinimumLicenseRequired: callbacks === null || callbacks === void 0 ? void 0 : callbacks.hasMinimumLicenseRequired,
        activeProduct: context === null || context === void 0 ? void 0 : context.activeProduct
      });
    }
    if (['in', 'not in'].includes(node.name)) {
      var _callbacks$getByType2, _context$columns;
      // // command ... a in ( <here> )
      // return { type: 'list' as const, command, node, option, containingFunction };
      return await getListArgsSuggestions(innerText, ast, (_callbacks$getByType2 = callbacks === null || callbacks === void 0 ? void 0 : callbacks.getByType) !== null && _callbacks$getByType2 !== void 0 ? _callbacks$getByType2 : () => Promise.resolve([]), (_context$columns = context === null || context === void 0 ? void 0 : context.columns) !== null && _context$columns !== void 0 ? _context$columns : new Map(), cursorPosition !== null && cursorPosition !== void 0 ? cursorPosition : 0, callbacks === null || callbacks === void 0 ? void 0 : callbacks.hasMinimumLicenseRequired, context === null || context === void 0 ? void 0 : context.activeProduct);
    }
    if (isNotEnrichClauseAssigment(node, command) && !isOperator(node)) {
      var _callbacks$getByType3;
      // command ... fn( <here> )
      return await getFunctionArgsSuggestions(innerText, ast, (_callbacks$getByType3 = callbacks === null || callbacks === void 0 ? void 0 : callbacks.getByType) !== null && _callbacks$getByType3 !== void 0 ? _callbacks$getByType3 : () => Promise.resolve([]), query, cursorPosition !== null && cursorPosition !== void 0 ? cursorPosition : 0, context, callbacks === null || callbacks === void 0 ? void 0 : callbacks.hasMinimumLicenseRequired);
    }
  }
  return undefined;
};
exports.getInsideFunctionsSuggestions = getInsideFunctionsSuggestions;