"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.suggestionUnion = exports.suggestionIntersection = exports.suggestFields = exports.getPosition = exports.getLookupFields = exports.getFullCommandMnemonics = exports.getFieldSuggestions = void 0;
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _helpers = require("../../../definitions/utils/autocomplete/helpers");
var _utils = require("../../../definitions/utils");
var _is = require("../../../ast/is");
var _shared = require("../../../definitions/utils/shared");
var mutate = _interopRequireWildcard(require("../../../mutate"));
var _leaf_printer = require("../../../pretty_print/leaf_printer");
var _complete_items = require("../../complete_items");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const REGEX = /^(?<type>\w+((?<after_type>\s+((?<mnemonic>(JOIN|JOI|JO|J)((?<after_mnemonic>\s+((?<index>\S+((?<after_index>\s+(?<as>(AS|A))?(?<after_as>\s+(((?<alias>\S+)?(?<after_alias>\s+)?)?))?((?<on>(ON|O)((?<after_on>\s+(?<cond>[^\s])?)?))?))?))?))?))?))?))?/i;
const positions = ['cond', 'after_on', 'on', 'after_alias', 'alias', 'after_as', 'as', 'after_index', 'index', 'after_mnemonic', 'mnemonic', 'after_type', 'type'];
const getFullCommandMnemonics = command => {
  var _command$metadata$typ;
  const types = (_command$metadata$typ = command.metadata.types) !== null && _command$metadata$typ !== void 0 ? _command$metadata$typ : [];
  if (!types.length) {
    return [[command.name, command.metadata.description]];
  }
  return types.map(type => {
    var _type$description;
    return [`${type.name.toUpperCase()} ${command.name.toUpperCase()}`, (_type$description = type.description) !== null && _type$description !== void 0 ? _type$description : command.metadata.description];
  });
};

// facilitates fast checks for the existence of fields in the lookup index
// by caching the fields of the last lookup index pattern
exports.getFullCommandMnemonics = getFullCommandMnemonics;
const lookupIndexFieldSet = {
  set: new Set(),
  key: ''
};
const getLookupFields = async (command, getColumnsForQuery, context) => {
  if (!context) {
    return [];
  }
  const summary = mutate.commands.join.summarizeCommand(command);
  const joinIndexPattern = _leaf_printer.LeafPrinter.print(summary.target.index);
  const columns = await getColumnsForQuery(`FROM ${joinIndexPattern}`);
  if (lookupIndexFieldSet.key !== joinIndexPattern) {
    lookupIndexFieldSet.set = new Set(columns.map(c => c.name));
    lookupIndexFieldSet.key = joinIndexPattern;
  }
  return columns;
};
exports.getLookupFields = getLookupFields;
const getFieldSuggestions = async (command, getColumnsByType, getColumnsForQuery, context) => {
  if (!context) {
    return {
      suggestions: [],
      lookupIndexFieldExists: () => false
    };
  }
  const onOption = command.args.find(arg => !Array.isArray(arg) && arg.name === 'on');
  const ignoredFields = onOption.args.map(arg => (0, _is.isColumn)(arg) ? arg.parts.join('.') : '');
  const [lookupIndexFields, sourceFields] = await Promise.all([getLookupFields(command, getColumnsForQuery, context), getColumnsByType(['any'], ignoredFields, {
    advanceCursor: false,
    openSuggestions: true
  })]);
  const joinFields = (0, _utils.buildFieldsDefinitionsWithMetadata)(lookupIndexFields.filter(f => !ignoredFields.includes(f.name)), [], {
    supportsControls: false
  }, // Controls are being added as part of the sourceFields, no need to add them again as joinFields.
  context === null || context === void 0 ? void 0 : context.variables);
  const intersection = suggestionIntersection(joinFields, sourceFields);
  const union = suggestionUnion(sourceFields, joinFields);
  for (const commonField of intersection) {
    commonField.sortText = '1';
    commonField.documentation = {
      value: _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.join.sharedField', {
        defaultMessage: 'Field shared between the source and the lookup index'
      })
    };
    let detail = commonField.detail || '';
    if (detail) {
      detail += ' ';
    }
    detail += _i18n.i18n.translate('kbn-esql-ast.esql.autocomplete.join.commonFieldNote', {
      defaultMessage: '(common field)'
    });
    commonField.detail = detail;
  }
  return {
    suggestions: (0, _lodash.uniqBy)([...intersection, ...union], 'label'),
    lookupIndexFieldExists: field => lookupIndexFieldSet.set.has((0, _shared.unescapeColumnName)(field))
  };
};
exports.getFieldSuggestions = getFieldSuggestions;
const suggestFields = async (innerText, command, getColumnsByType, getColumnsForQuery, context) => {
  if (!context) {
    return [];
  }
  const {
    suggestions: fieldSuggestions,
    lookupIndexFieldExists
  } = await getFieldSuggestions(command, getColumnsByType,
  // this type cast is ok because getFieldSuggestions only ever fetches columns
  // from a bare FROM clause, so they will always be fields, not user-defined columns
  getColumnsForQuery, context);
  return (0, _helpers.handleFragment)(innerText, fragment => (0, _helpers.columnExists)(fragment, context) || lookupIndexFieldExists(fragment), (_fragment, rangeToReplace) => {
    // fie<suggest>
    return fieldSuggestions.map(suggestion => {
      return (0, _helpers.withAutoSuggest)({
        ...suggestion,
        text: suggestion.text,
        rangeToReplace
      });
    });
  }, (fragment, rangeToReplace) => {
    // field<suggest>
    const finalSuggestions = [{
      ..._complete_items.pipeCompleteItem,
      text: ' | '
    }];
    // when we fix the editor marker, this should probably be checked against 0 instead of 1
    // this is because the last field in the AST is currently getting removed (because it contains
    // the editor marker) so it is not included in the ignored list which is used to filter out
    // existing fields above.
    if (fieldSuggestions.length > 1) finalSuggestions.push({
      ..._complete_items.commaCompleteItem,
      text: ', '
    });
    return finalSuggestions.map(s => (0, _helpers.withAutoSuggest)({
      ...s,
      filterText: fragment,
      text: fragment + s.text,
      rangeToReplace
    }));
  });
};

/**
 * Returns the static position, or `cond` if the caret is in the `<conditions>`
 * part of the command, in which case further parsing is needed.
 */
exports.suggestFields = suggestFields;
const getStaticPosition = text => {
  const match = text.match(REGEX);
  if (!match || !match.groups) {
    return 'none';
  }
  let pos = 'cond';
  for (const position of positions) {
    if (match.groups[position]) {
      pos = position;
      break;
    }
  }
  return pos;
};
const getPosition = text => {
  const pos0 = getStaticPosition(text);
  const pos = pos0 === 'cond' ? 'condition' : pos0;
  return {
    pos,
    type: ''
  };
};
exports.getPosition = getPosition;
const suggestionIntersection = (suggestions1, suggestions2) => {
  const labels1 = new Set();
  const intersection = [];
  for (const suggestion1 of suggestions1) {
    labels1.add(suggestion1.label);
  }
  for (const suggestion2 of suggestions2) {
    if (labels1.has(suggestion2.label)) {
      intersection.push({
        ...suggestion2
      });
    }
  }
  return intersection;
};
exports.suggestionIntersection = suggestionIntersection;
const suggestionUnion = (suggestions1, suggestions2) => {
  const labels = new Set();
  const union = [];
  for (const suggestion of suggestions1) {
    const label = suggestion.label;
    if (!labels.has(label)) {
      union.push(suggestion);
      labels.add(label);
    }
  }
  for (const suggestion of suggestions2) {
    const label = suggestion.label;
    if (!labels.has(label)) {
      union.push(suggestion);
      labels.add(label);
    }
  }
  return union;
};
exports.suggestionUnion = suggestionUnion;