"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.findAttackDiscoveriesInternalRoute = void 0;
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _zodHelpers = require("@kbn/zod-helpers");
var _helpers = require("../../../helpers");
var _build_response = require("../../../../lib/build_response");
var _index_privileges = require("../../helpers/index_privileges");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const findAttackDiscoveriesInternalRoute = router => {
  router.versioned.get({
    access: 'internal',
    path: _elasticAssistantCommon.ATTACK_DISCOVERY_INTERNAL_FIND,
    security: {
      authz: {
        requiredPrivileges: ['elasticAssistant']
      }
    }
  }).addVersion({
    version: _elasticAssistantCommon.API_VERSIONS.internal.v1,
    validate: {
      request: {
        query: (0, _zodHelpers.buildRouteValidationWithZod)(_elasticAssistantCommon.AttackDiscoveryFindInternalRequestQuery)
      },
      response: {
        200: {
          body: {
            custom: (0, _zodHelpers.buildRouteValidationWithZod)(_elasticAssistantCommon.AttackDiscoveryFindInternalResponse)
          }
        }
      }
    }
  }, async (context, request, response) => {
    const ctx = await context.resolve(['core', 'elasticAssistant', 'licensing']);
    const resp = (0, _build_response.buildResponse)(response);
    const assistantContext = await context.elasticAssistant;
    const logger = assistantContext.logger;

    // Perform license and authenticated user checks:
    const checkResponse = await (0, _helpers.performChecks)({
      context: ctx,
      request,
      response
    });
    if (!checkResponse.isSuccess) {
      return checkResponse.response;
    }

    // Perform alerts access check
    const privilegesCheckResponse = await (0, _index_privileges.hasReadAttackDiscoveryAlertsPrivileges)({
      context: ctx,
      response
    });
    if (!privilegesCheckResponse.isSuccess) {
      return privilegesCheckResponse.response;
    }
    try {
      var _query$include_unique;
      const {
        query
      } = request;
      const dataClient = await assistantContext.getAttackDiscoveryDataClient();
      if (!dataClient) {
        return resp.error({
          body: `Attack discovery data client not initialized`,
          statusCode: 500
        });
      }
      const currentUser = await checkResponse.currentUser;

      // get an Elasticsearch client for the authenticated user:
      const esClient = (await context.core).elasticsearch.client.asCurrentUser;
      const result = await dataClient.findAttackDiscoveryAlerts({
        authenticatedUser: currentUser,
        esClient,
        findAttackDiscoveryAlertsParams: {
          alertIds: query.alert_ids,
          includeUniqueAlertIds: (_query$include_unique = query.include_unique_alert_ids) !== null && _query$include_unique !== void 0 ? _query$include_unique : false,
          ids: query.ids,
          search: query.search,
          shared: query.shared,
          status: query.status,
          connectorNames: query.connector_names,
          start: query.start,
          enableFieldRendering: true,
          // always true for internal apis
          end: query.end,
          page: query.page,
          perPage: query.per_page,
          sortField: query.sort_field,
          sortOrder: query.sort_order,
          withReplacements: false // always false for internal apis
        },
        logger
      });
      return response.ok({
        body: {
          ...result,
          data: result.data.map(_elasticAssistantCommon.transformAttackDiscoveryAlertFromApi) // required by legacy internal routes
        }
      });
    } catch (err) {
      logger.error(err);
      const error = (0, _securitysolutionEsUtils.transformError)(err);
      return resp.error({
        body: error.message,
        statusCode: error.statusCode
      });
    }
  });
};
exports.findAttackDiscoveriesInternalRoute = findAttackDiscoveriesInternalRoute;