"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isValidExampleOutput = exports.customIncompatibleAntivirusEvaluator = void 0;
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const isValidExampleOutput = output => {
  // Check if output is an object and has the expected structure. It's defined in LangSmith, hence needs validation.
  const isStringArray = arr => Array.isArray(arr) && arr.every(item => typeof item === 'string');
  return output && Array.isArray(output.results) && output.results.every(res => typeof res.name === 'string' && isStringArray(res.requiredPaths) && isStringArray(res.optionalPaths) && isStringArray(res.excludedPaths));
};
exports.isValidExampleOutput = isValidExampleOutput;
const failWithComment = comment => ({
  key: 'correct',
  score: 0,
  comment
});
const customIncompatibleAntivirusEvaluator = (run, example) => {
  var _run$outputs$insights, _run$outputs;
  const rawOutput = example === null || example === void 0 ? void 0 : example.outputs;
  if (!isValidExampleOutput(rawOutput)) {
    return failWithComment(_constants.EVALUATOR_ERRORS.INVALID_OUTPUT_STRUCTURE);
  }
  const {
    results: requirements
  } = rawOutput;
  const insights = (_run$outputs$insights = (_run$outputs = run.outputs) === null || _run$outputs === void 0 ? void 0 : _run$outputs.insights) !== null && _run$outputs$insights !== void 0 ? _run$outputs$insights : [];
  if (!Array.isArray(insights) || insights.length === 0) {
    return failWithComment(_constants.EVALUATOR_ERRORS.NO_RESULTS);
  }
  const failedChecks = [];
  let totalChecks = 0;

  // Check: group count matches requirement count
  totalChecks += 1;
  if (insights.length !== requirements.length) {
    failedChecks.push({
      label: 'number of insight groups does not match number of requirements',
      details: [`insights: ${insights.length}`, `requirements: ${requirements.length}`]
    });
  }
  for (const req of requirements) {
    const label = `requirement "${req.name}"`;
    const matchedInsight = insights.find(insight => insight.group.toLowerCase().includes(req.name.toLowerCase()));
    totalChecks += 3;
    if (!matchedInsight) {
      failedChecks.push({
        label: `${label} did not match any insight group`
      });
    } else {
      const eventPaths = (matchedInsight.events || []).map(e => e.value);
      const requiredSet = new Set(req.requiredPaths);
      const excludedSet = new Set(req.excludedPaths);
      const allowedSet = new Set([...req.requiredPaths, ...req.optionalPaths]);
      const missingRequired = [...requiredSet].filter(p => !eventPaths.includes(p));
      if (missingRequired.length) {
        failedChecks.push({
          label: `${label} is missing required paths`,
          details: missingRequired
        });
      }
      const foundExcluded = eventPaths.filter(p => excludedSet.has(p));
      if (foundExcluded.length) {
        failedChecks.push({
          label: `${label} contains excluded paths`,
          details: foundExcluded
        });
      }
      const unexpected = eventPaths.filter(p => !allowedSet.has(p) && !excludedSet.has(p));
      if (unexpected.length) {
        failedChecks.push({
          label: `${label} contains unexpected paths`,
          details: unexpected
        });
      }
    }
  }
  const score = totalChecks === 0 ? 0 : Number((1 - failedChecks.length / totalChecks).toFixed(2));
  const comment = failedChecks.length ? `Failed checks:\n${failedChecks.map(c => {
    var _c$details;
    return (_c$details = c.details) !== null && _c$details !== void 0 && _c$details.length ? `${c.label}:\n  - ${c.details.join('\n  - ')}` : c.label;
  }).join('\n\n')}` : 'All checks passed';
  return {
    key: 'correct',
    score,
    comment
  };
};
exports.customIncompatibleAntivirusEvaluator = customIncompatibleAntivirusEvaluator;