"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createAttackDiscoveryAlerts = void 0;
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _fp = require("lodash/fp");
var _uuid = require("uuid");
var _transform_to_alert_documents = require("../transforms/transform_to_alert_documents");
var _get_created_document_ids = require("./get_created_document_ids");
var _get_created_attack_discovery_alerts = require("./get_created_attack_discovery_alerts");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const DEBUG_LOG_ID_PLACEHOLDER = `(uuid will replace missing ${_ruleDataUtils.ALERT_UUID})`;
const createAttackDiscoveryAlerts = async ({
  adhocAttackDiscoveryDataClient,
  authenticatedUser,
  createAttackDiscoveryAlertsParams,
  logger,
  spaceId
}) => {
  const attackDiscoveryAlertsIndex = adhocAttackDiscoveryDataClient.indexNameWithNamespace(spaceId);
  const readDataClient = adhocAttackDiscoveryDataClient.getReader({
    namespace: spaceId
  });
  const writeDataClient = await adhocAttackDiscoveryDataClient.getWriter({
    namespace: spaceId
  });
  const now = new Date();
  const alertDocuments = (0, _transform_to_alert_documents.transformToAlertDocuments)({
    authenticatedUser,
    createAttackDiscoveryAlertsParams,
    now,
    spaceId
  });
  if ((0, _fp.isEmpty)(alertDocuments)) {
    logger.debug(() => `No Attack discovery alerts to create for index ${attackDiscoveryAlertsIndex} in createAttackDiscoveryAlerts`);
    return [];
  }
  const alertIds = alertDocuments.map(alertDocument => {
    var _alertDocument$ALERT_;
    return (_alertDocument$ALERT_ = alertDocument[_ruleDataUtils.ALERT_UUID]) !== null && _alertDocument$ALERT_ !== void 0 ? _alertDocument$ALERT_ : DEBUG_LOG_ID_PLACEHOLDER;
  });
  try {
    logger.debug(() => `Creating Attack discovery alerts in index ${attackDiscoveryAlertsIndex} with alert ids: ${alertIds.join(', ')}`);
    const body = alertDocuments.flatMap(alertDocument => {
      var _alertDocument$ALERT_2;
      return [{
        create: {
          _id: (_alertDocument$ALERT_2 = alertDocument[_ruleDataUtils.ALERT_UUID]) !== null && _alertDocument$ALERT_2 !== void 0 ? _alertDocument$ALERT_2 : (0, _uuid.v4)()
        }
      }, alertDocument];
    });
    const resp = await writeDataClient.bulk({
      body,
      refresh: true
    });
    const bulkResponse = resp === null || resp === void 0 ? void 0 : resp.body;
    if (!bulkResponse) {
      logger.info(`Rule data client returned undefined as a result of the bulk operation.`);
      return [];
    }
    const createdDocumentIds = (0, _get_created_document_ids.getCreatedDocumentIds)(bulkResponse);
    if (bulkResponse.errors) {
      const errorDetails = bulkResponse.items.flatMap(item => {
        var _item$create, _item$create2;
        const error = (_item$create = item.create) === null || _item$create === void 0 ? void 0 : _item$create.error;
        if (error == null) {
          return [];
        }
        const id = ((_item$create2 = item.create) === null || _item$create2 === void 0 ? void 0 : _item$create2._id) != null ? ` id: ${item.create._id}` : '';
        const details = `\nError bulk inserting attack discovery alert${id} ${error.reason}`;
        return [details];
      });
      const allErrorDetails = errorDetails.join(', ');
      throw new Error(`Failed to bulk insert Attack discovery alerts ${allErrorDetails}`);
    }
    logger.debug(() => `Created Attack discovery alerts in index ${attackDiscoveryAlertsIndex} with document ids: ${createdDocumentIds.join(', ')}`);
    const {
      enableFieldRendering,
      withReplacements
    } = createAttackDiscoveryAlertsParams;
    return (0, _get_created_attack_discovery_alerts.getCreatedAttackDiscoveryAlerts)({
      attackDiscoveryAlertsIndex,
      createdDocumentIds,
      enableFieldRendering,
      logger,
      readDataClient,
      withReplacements
    });
  } catch (err) {
    logger.error(`Error creating Attack discovery alerts in index ${attackDiscoveryAlertsIndex}: ${err} with alert ids: ${alertIds.join(', ')}`);
    throw err;
  }
};
exports.createAttackDiscoveryAlerts = createAttackDiscoveryAlerts;