"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getChartPreview = getChartPreview;
exports.getFilteredBarSeries = void 0;
var _extract_key = require("../extract_key");
var _types = require("../types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const NUM_SERIES = 5;
const DEFAULT_GROUPS = 1000;
const getFilteredBarSeries = barSeries => {
  const sortedSeries = barSeries.sort((a, b) => {
    const aMax = Math.max(...a.data.map(point => point.y));
    const bMax = Math.max(...b.data.map(point => point.y));
    return bMax - aMax;
  });
  return sortedSeries.slice(0, NUM_SERIES);
};
exports.getFilteredBarSeries = getFilteredBarSeries;
async function getChartPreview({
  esClient,
  index,
  groupBy,
  start,
  end,
  interval
}) {
  var _esResult$aggregation, _esResult$aggregation2, _esResult$aggregation3;
  const bool = {
    filter: [{
      range: {
        '@timestamp': {
          gte: start,
          lte: end,
          format: 'epoch_millis'
        }
      }
    }]
  };
  const aggs = {
    ...(groupBy.length > 0 ? {
      series: {
        ...(groupBy.length === 1 ? {
          terms: {
            field: groupBy[0],
            size: DEFAULT_GROUPS,
            order: {
              _count: 'desc'
            }
          }
        } : {
          multi_terms: {
            terms: groupBy.map(field => ({
              field,
              missing: _types.MISSING_VALUE
            })),
            size: DEFAULT_GROUPS,
            order: {
              _count: 'desc'
            }
          }
        }),
        aggs: {
          timeseries: {
            date_histogram: {
              field: '@timestamp',
              fixed_interval: interval,
              extended_bounds: {
                min: start,
                max: end
              }
            },
            aggs: {
              ignored_fields: {
                filter: {
                  exists: {
                    field: '_ignored'
                  }
                }
              }
            }
          }
        }
      }
    } : {})
  };
  const esResult = await esClient.search({
    index,
    track_total_hits: false,
    size: 0,
    query: {
      bool
    },
    aggs
  });
  if (!esResult.aggregations) {
    return {
      series: [],
      totalGroups: 0
    };
  }
  const seriesBuckets = (_esResult$aggregation = (_esResult$aggregation2 = esResult.aggregations) === null || _esResult$aggregation2 === void 0 ? void 0 : (_esResult$aggregation3 = _esResult$aggregation2.series) === null || _esResult$aggregation3 === void 0 ? void 0 : _esResult$aggregation3.buckets) !== null && _esResult$aggregation !== void 0 ? _esResult$aggregation : [];
  const seriesDataMap = seriesBuckets.reduce((acc, bucket) => {
    const bucketKey = (0, _extract_key.extractKey)({
      groupBy,
      bucketKey: Array.isArray(bucket.key) ? bucket.key : [bucket.key]
    });
    const timeSeriesBuckets = bucket.timeseries;
    const timeseries = timeSeriesBuckets.buckets;
    timeseries.forEach(timeseriesBucket => {
      var _timeseriesBucket$doc, _doc_count, _timeseriesBucket$ign;
      const x = timeseriesBucket.key;
      const totalCount = (_timeseriesBucket$doc = timeseriesBucket.doc_count) !== null && _timeseriesBucket$doc !== void 0 ? _timeseriesBucket$doc : 0;
      const ignoredCount = (_doc_count = (_timeseriesBucket$ign = timeseriesBucket.ignored_fields) === null || _timeseriesBucket$ign === void 0 ? void 0 : _timeseriesBucket$ign.doc_count) !== null && _doc_count !== void 0 ? _doc_count : 0;
      if (acc[bucketKey.join(',')]) {
        acc[bucketKey.join(',')].push({
          x,
          totalCount,
          ignoredCount
        });
      } else {
        acc[bucketKey.join(',')] = [{
          x,
          totalCount,
          ignoredCount
        }];
      }
    });
    return acc;
  }, {});
  const series = Object.keys(seriesDataMap).map(key => ({
    name: key,
    data: Array.from(seriesDataMap[key].reduce((map, curr) => {
      if (!map.has(curr.x)) map.set(curr.x, {
        ...curr
      });else {
        map.get(curr.x).totalCount += curr.totalCount;
        map.get(curr.x).ignoredCount += curr.ignoredCount;
      }
      return map;
    }, new Map()).values()).map(item => ({
      x: item.x,
      y: item.totalCount ? item.ignoredCount / item.totalCount * 100 : 0
    }))
  }));
  const filteredSeries = getFilteredBarSeries(series);
  return {
    series: filteredSeries,
    totalGroups: series.length
  };
}