"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.statusResponse = void 0;
var _configSchema = require("@kbn/config-schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const serviceStatusLevelId = () => _configSchema.schema.oneOf([_configSchema.schema.literal('available'), _configSchema.schema.literal('degraded'), _configSchema.schema.literal('unavailable'), _configSchema.schema.literal('critical')], {
  meta: {
    description: 'Service status levels as human and machine readable values.'
  }
});
const statusInfoServiceStatus = () => _configSchema.schema.object({
  level: serviceStatusLevelId(),
  summary: _configSchema.schema.string({
    meta: {
      description: 'A human readable summary of the service status.'
    }
  }),
  detail: _configSchema.schema.maybe(_configSchema.schema.string({
    meta: {
      description: 'Human readable detail of the service status.'
    }
  })),
  documentationUrl: _configSchema.schema.maybe(_configSchema.schema.string({
    meta: {
      description: 'A URL to further documentation regarding this service.'
    }
  })),
  meta: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.any(), {
    meta: {
      description: 'An unstructured set of extra metadata about this service.'
    }
  })
});
const statusInfoCoreStatus = () => _configSchema.schema.object({
  elasticsearch: statusInfoServiceStatus(),
  http: _configSchema.schema.maybe(statusInfoServiceStatus()),
  savedObjects: statusInfoServiceStatus()
}, {
  meta: {
    description: 'Statuses of core Kibana services.'
  }
});

/** Only include a subset of fields for OAS documentation, for now */
const serverMetrics = () => _configSchema.schema.object({
  elasticsearch_client: _configSchema.schema.object({
    totalActiveSockets: _configSchema.schema.number({
      meta: {
        description: 'Count of network sockets currently in use.'
      }
    }),
    totalIdleSockets: _configSchema.schema.number({
      meta: {
        description: 'Count of network sockets currently idle.'
      }
    }),
    totalQueuedRequests: _configSchema.schema.number({
      meta: {
        description: 'Count of requests not yet assigned to sockets.'
      }
    })
  }, {
    meta: {
      description: `Current network metrics of Kibana's Elasticsearch client.`
    }
  }),
  last_updated: _configSchema.schema.string({
    meta: {
      description: 'The time metrics were collected.'
    }
  }),
  collection_interval_in_millis: _configSchema.schema.number({
    meta: {
      description: 'The interval at which metrics should be collected.'
    }
  })
}, {
  meta: {
    description: 'Metric groups collected by Kibana.'
  }
});
const buildFlavour = () => _configSchema.schema.oneOf([_configSchema.schema.literal('serverless'), _configSchema.schema.literal('traditional')], {
  meta: {
    description: 'The build flavour determines configuration and behavior of Kibana. On premise users will almost always run the "traditional" flavour, while other flavours are reserved for Elastic-specific use cases.'
  }
});
const serverVersion = () => _configSchema.schema.object({
  number: _configSchema.schema.string({
    meta: {
      description: 'A semantic version number.'
    }
  }),
  build_hash: _configSchema.schema.string({
    meta: {
      description: 'A unique hash value representing the git commit of this Kibana build.'
    }
  }),
  build_number: _configSchema.schema.number({
    meta: {
      description: 'A monotonically increasing number, each subsequent build will have a higher number.'
    }
  }),
  build_snapshot: _configSchema.schema.boolean({
    meta: {
      description: 'Whether this build is a snapshot build.'
    }
  }),
  build_flavor: buildFlavour(),
  build_date: _configSchema.schema.string({
    meta: {
      description: 'The date and time of this build.'
    }
  })
});
const statusInfo = () => _configSchema.schema.object({
  overall: statusInfoServiceStatus(),
  core: statusInfoCoreStatus(),
  plugins: _configSchema.schema.recordOf(_configSchema.schema.string(), statusInfoServiceStatus(), {
    meta: {
      description: 'A dynamic mapping of plugin ID to plugin status.'
    }
  })
});

/** Excluding metrics for brevity, for now */
const fullStatusResponse = () => _configSchema.schema.object({
  name: _configSchema.schema.string({
    meta: {
      description: 'Kibana instance name.'
    }
  }),
  uuid: _configSchema.schema.string({
    meta: {
      description: 'Unique, generated Kibana instance UUID. This UUID should persist even if the Kibana process restarts.'
    }
  }),
  version: serverVersion(),
  status: statusInfo(),
  metrics: serverMetrics()
}, {
  meta: {
    id: 'core_status_response',
    description: `Kibana's operational status as well as a detailed breakdown of plugin statuses indication of various loads (like event loop utilization and network traffic) at time of request.`
  }
});
const redactedStatusResponse = () => _configSchema.schema.object({
  status: _configSchema.schema.object({
    overall: _configSchema.schema.object({
      level: serviceStatusLevelId()
    })
  })
}, {
  meta: {
    id: 'core_status_redactedResponse',
    description: `A minimal representation of Kibana's operational status.`
  }
});

/** Lazily load this schema */
const statusResponse = () => _configSchema.schema.oneOf([fullStatusResponse(), redactedStatusResponse()], {
  meta: {
    description: `Kibana's operational status. A minimal response is sent for unauthorized users.`
  }
});
exports.statusResponse = statusResponse;