"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.OsMetricsCollector = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _os = _interopRequireDefault(require("os"));
var _getos = _interopRequireDefault(require("getos"));
var _util = require("util");
var _api = require("@opentelemetry/api");
var _cgroup = require("./cgroup");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const getos = (0, _util.promisify)(_getos.default);
class OsMetricsCollector {
  constructor(options) {
    (0, _defineProperty2.default)(this, "cgroupCollector", void 0);
    (0, _defineProperty2.default)(this, "log", void 0);
    this.cgroupCollector = new _cgroup.OsCgroupMetricsCollector({
      ...options,
      logger: options.logger.get('cgroup')
    });
    this.log = options.logger;
  }
  async collect() {
    const platform = _os.default.platform();
    const load = _os.default.loadavg();
    const osMetrics = {
      platform,
      platformRelease: `${platform}-${_os.default.release()}`,
      load: {
        '1m': load[0],
        '5m': load[1],
        '15m': load[2]
      },
      memory: {
        total_in_bytes: _os.default.totalmem(),
        free_in_bytes: _os.default.freemem(),
        used_in_bytes: _os.default.totalmem() - _os.default.freemem()
      },
      uptime_in_millis: _os.default.uptime() * 1000,
      ...(await this.getDistroStats(platform)),
      ...(await this.cgroupCollector.collect())
    };
    return osMetrics;
  }
  reset() {}
  registerMetrics() {
    const meter = _api.metrics.getMeter('kibana.os');

    // https://opentelemetry.io/docs/specs/semconv/registry/attributes/os/#operating-system-attributes
    const platform = _os.default.platform();
    const attributes = {
      'os.type': _os.default.platform(),
      'os.version': _os.default.release()
    };
    this.getDistroStats(platform).then(({
      distroRelease
    }) => {
      attributes['os.name'] = distroRelease;
    }).catch(err => {
      this.log.error(`Could not determine distro name due to error: [${err.toString()}]`);
    });

    // https://opentelemetry.io/docs/specs/semconv/system/system-metrics/
    meter.createObservableGauge('system.uptime', {
      description: 'The time the system has been running.',
      unit: 's',
      valueType: _api.ValueType.DOUBLE
    }).addCallback(result => {
      result.observe(_os.default.uptime(), attributes);
    });
    meter.createObservableUpDownCounter('system.cpu.logical.count', {
      description: 'The number of logical CPUs.',
      unit: '1',
      valueType: _api.ValueType.INT
    }).addCallback(result => {
      result.observe(_os.default.cpus().length, attributes);
    });
    meter.createObservableGauge('system.cpu.load', {
      description: 'The CPU load average.',
      unit: '1',
      valueType: _api.ValueType.DOUBLE
    }).addCallback(result => {
      const load = _os.default.loadavg();
      result.observe(load[0], {
        ...attributes,
        'system.cpu.load.window': '1m'
      });
      result.observe(load[1], {
        ...attributes,
        'system.cpu.load.window': '5m'
      });
      result.observe(load[2], {
        ...attributes,
        'system.cpu.load.window': '15m'
      });
    });
    const memoryMetrics = {
      limit: meter.createObservableUpDownCounter('system.memory.limit', {
        description: 'Total virtual memory available in the system.',
        unit: 'By',
        valueType: _api.ValueType.INT
      }),
      usage: meter.createObservableUpDownCounter('system.memory.usage', {
        description: 'Reports memory in use by state.',
        unit: 'By',
        valueType: _api.ValueType.INT
      })
    };
    meter.addBatchObservableCallback(result => {
      const limit = _os.default.totalmem();
      const free = _os.default.freemem();
      const used = limit - free;
      result.observe(memoryMetrics.limit, limit, attributes);
      result.observe(memoryMetrics.usage, used, {
        ...attributes,
        'system.memory.state': 'used'
      });
      result.observe(memoryMetrics.usage, free, {
        ...attributes,
        'system.memory.state': 'free'
      });
    }, Object.values(memoryMetrics));
    const cgroupMetrics = {
      accountingUsage: meter.createObservableGauge('system.cgroup.cpuacct.usage', {
        description: 'The amount of time in nanoseconds that the cgroup has been scheduled in user mode.',
        unit: 'ns',
        valueType: _api.ValueType.INT
      }),
      cfsPeriod: meter.createObservableGauge('system.cgroup.cfs.period', {
        description: 'OS CPU cgroup: the length of the cfs period in microseconds',
        unit: 'us',
        valueType: _api.ValueType.INT
      }),
      cfsQuota: meter.createObservableGauge('system.cgroup.cfs.quota', {
        description: 'OS CPU cgroup: total available run-time within a cfs period in microseconds',
        unit: 'us',
        valueType: _api.ValueType.INT
      }),
      cfsElapsed: meter.createObservableGauge('system.cgroup.cfs.elapsed', {
        description: 'OS CPU cgroup: number of cfs periods that elapsed',
        unit: '1',
        valueType: _api.ValueType.INT
      }),
      cfsThrottled: meter.createObservableGauge('system.cgroup.cfs.throttled', {
        description: 'OS CPU cgroup: number of times the cgroup has been throttled',
        unit: '1',
        valueType: _api.ValueType.INT
      }),
      cgroupThrottled: meter.createObservableGauge('system.cgroup.throttled.time', {
        description: 'OS CPU cgroup: total amount of time the cgroup has been throttled for in nanoseconds',
        unit: 'ns',
        valueType: _api.ValueType.INT
      }),
      cgroupMemory: meter.createObservableGauge('system.cgroup.memory.usage', {
        description: 'OS CPU cgroup: total amount of memory currently being used by the cgroup and its descendants',
        unit: 'By',
        valueType: _api.ValueType.INT
      }),
      cgroupSwap: meter.createObservableGauge('system.cgroup.swap.usage', {
        description: 'OS CPU cgroup: total amount of swap currently being used by the cgroup and its descendants',
        unit: 'By',
        valueType: _api.ValueType.INT
      })
    };
    function observeMetricIfSet(result, metric, value, attrs) {
      if (value !== undefined) {
        result.observe(metric, value, attrs);
      }
    }
    meter.addBatchObservableCallback(async result => {
      var _collectedMetrics$cpu, _collectedMetrics$cpu2, _collectedMetrics$cpu3, _collectedMetrics$cpu4, _collectedMetrics$cpu5, _collectedMetrics$cpu6, _collectedMetrics$cpu7, _collectedMetrics$cgr, _collectedMetrics$cgr2;
      const collectedMetrics = await this.cgroupCollector.collect();
      const cgroupAttributes = {
        ...attributes,
        'system.cgroup.name': (_collectedMetrics$cpu = collectedMetrics.cpuacct) === null || _collectedMetrics$cpu === void 0 ? void 0 : _collectedMetrics$cpu.control_group
      };
      observeMetricIfSet(result, cgroupMetrics.accountingUsage, (_collectedMetrics$cpu2 = collectedMetrics.cpuacct) === null || _collectedMetrics$cpu2 === void 0 ? void 0 : _collectedMetrics$cpu2.usage_nanos, cgroupAttributes);
      observeMetricIfSet(result, cgroupMetrics.cfsPeriod, (_collectedMetrics$cpu3 = collectedMetrics.cpu) === null || _collectedMetrics$cpu3 === void 0 ? void 0 : _collectedMetrics$cpu3.cfs_period_micros, cgroupAttributes);
      observeMetricIfSet(result, cgroupMetrics.cfsQuota, (_collectedMetrics$cpu4 = collectedMetrics.cpu) === null || _collectedMetrics$cpu4 === void 0 ? void 0 : _collectedMetrics$cpu4.cfs_quota_micros, cgroupAttributes);
      observeMetricIfSet(result, cgroupMetrics.cfsElapsed, (_collectedMetrics$cpu5 = collectedMetrics.cpu) === null || _collectedMetrics$cpu5 === void 0 ? void 0 : _collectedMetrics$cpu5.stat.number_of_elapsed_periods, cgroupAttributes);
      observeMetricIfSet(result, cgroupMetrics.cfsThrottled, (_collectedMetrics$cpu6 = collectedMetrics.cpu) === null || _collectedMetrics$cpu6 === void 0 ? void 0 : _collectedMetrics$cpu6.stat.number_of_times_throttled, cgroupAttributes);
      observeMetricIfSet(result, cgroupMetrics.cgroupThrottled, (_collectedMetrics$cpu7 = collectedMetrics.cpu) === null || _collectedMetrics$cpu7 === void 0 ? void 0 : _collectedMetrics$cpu7.stat.time_throttled_nanos, cgroupAttributes);
      observeMetricIfSet(result, cgroupMetrics.cgroupMemory, (_collectedMetrics$cgr = collectedMetrics.cgroup_memory) === null || _collectedMetrics$cgr === void 0 ? void 0 : _collectedMetrics$cgr.current_in_bytes, cgroupAttributes);
      observeMetricIfSet(result, cgroupMetrics.cgroupSwap, (_collectedMetrics$cgr2 = collectedMetrics.cgroup_memory) === null || _collectedMetrics$cgr2 === void 0 ? void 0 : _collectedMetrics$cgr2.swap_current_in_bytes, cgroupAttributes);
    }, Object.values(cgroupMetrics));
  }
  async getDistroStats(platform) {
    if (platform === 'linux') {
      try {
        const distro = await getos();
        // getos values can sometimes contain newline characters
        const dist = removeNewlines(distro.dist);
        const release = removeNewlines(distro.release);
        return {
          distro: dist,
          distroRelease: `${dist}-${release}`
        };
      } catch (e) {
        // ignore errors
      }
    }
    return {};
  }
}
exports.OsMetricsCollector = OsMetricsCollector;
const removeNewlines = str => str.replace(/[\n]/g, '');