"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ExternalUrlService = void 0;
var _cryptoBrowser = require("@kbn/crypto-browser");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

function* getHostHashes(actualHost) {
  yield new _cryptoBrowser.Sha256().update(actualHost, 'utf8').digest('hex');
  let host = actualHost.substr(actualHost.indexOf('.') + 1);
  while (host) {
    yield new _cryptoBrowser.Sha256().update(host, 'utf8').digest('hex');
    if (host.indexOf('.') === -1) {
      break;
    }
    host = host.substr(host.indexOf('.') + 1);
  }
}
const isHostMatch = (actualHost, ruleHostHash) => {
  // If the host contains a `[`, then it's likely an IPv6 address. Otherwise, append a `.` if it doesn't already contain one
  const hostToHash = !actualHost.includes('[') && !actualHost.endsWith('.') ? `${actualHost}.` : actualHost;
  for (const hash of getHostHashes(hostToHash)) {
    if (hash === ruleHostHash) {
      return true;
    }
  }
  return false;
};
const isProtocolMatch = (actualProtocol, ruleProtocol) => {
  return normalizeProtocol(actualProtocol) === normalizeProtocol(ruleProtocol);
};
function normalizeProtocol(protocol) {
  return protocol.endsWith(':') ? protocol.slice(0, -1).toLowerCase() : protocol.toLowerCase();
}
const createIsInternalUrlValidation = (location, serverBasePath) => {
  return function isInternallUrl(next) {
    const base = new URL(location.href);
    const url = new URL(next, base);
    return url.origin === base.origin && (!serverBasePath || url.pathname.startsWith(`${serverBasePath}/`));
  };
};
const createExternalUrlValidation = (rules, location, serverBasePath) => {
  const isInternalUrl = createIsInternalUrlValidation(location, serverBasePath);
  return function validateExternalUrl(next) {
    const base = new URL(location.href);
    const url = new URL(next, base);
    if (isInternalUrl(next)) {
      return url;
    }
    let allowed = null;
    rules.forEach(rule => {
      const hostMatch = rule.host ? isHostMatch(url.hostname || '', rule.host) : true;
      const protocolMatch = rule.protocol ? isProtocolMatch(url.protocol, rule.protocol) : true;
      const isRuleMatch = hostMatch && protocolMatch;
      if (isRuleMatch && allowed !== false) {
        allowed = rule.allow;
      }
    });
    return allowed === true ? url : null;
  };
};
class ExternalUrlService {
  setup({
    injectedMetadata,
    location
  }) {
    const serverBasePath = injectedMetadata.getServerBasePath();
    const {
      policy
    } = injectedMetadata.getExternalUrlConfig();
    return {
      isInternalUrl: createIsInternalUrlValidation(location, serverBasePath),
      validateUrl: createExternalUrlValidation(policy, location, serverBasePath)
    };
  }
  start() {}
  stop() {}
}
exports.ExternalUrlService = ExternalUrlService;