"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getPoliciesToDelete = exports.getPoliciesByConnectorId = exports.getConnectorsToDeploy = exports.getConnectorPolicyId = exports.AgentlessConnectorsInfraService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _common = require("@kbn/fleet-plugin/common");
var _searchConnectors = require("@kbn/search-connectors");
var _packages = require("@kbn/fleet-plugin/server/services/epm/packages");
var _common2 = require("@kbn/spaces-plugin/common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// Agent metadata is only returned when there is an agent associated with the policy

const connectorsInputName = 'connectors-py';
const pkgName = 'elastic_connectors';
const pkgTitle = 'Elastic Connectors';
class AgentlessConnectorsInfraService {
  constructor(soClient, esClient, packagePolicyService, agentPolicyService, agentService, logger) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "soClient", void 0);
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "packagePolicyService", void 0);
    (0, _defineProperty2.default)(this, "agentPolicyService", void 0);
    (0, _defineProperty2.default)(this, "agentService", void 0);
    (0, _defineProperty2.default)(this, "getNativeConnectors", async () => {
      this.logger.debug(`Fetching all connectors and filtering only to native`);
      const nativeConnectors = [];
      const allConnectors = await (0, _searchConnectors.fetchConnectors)(this.esClient, undefined, undefined, undefined, true // includeDeleted
      );
      for (const connector of allConnectors) {
        if (connector.is_native && connector.service_type != null) {
          if (_searchConnectors.NATIVE_CONNECTOR_DEFINITIONS[connector.service_type] == null) {
            this.logger.debug(`Skipping connector ${connector.id}: unsupported service type ${connector.service_type}`);
            continue;
          }
          nativeConnectors.push({
            id: connector.id,
            name: connector.name,
            service_type: connector.service_type,
            is_deleted: !!connector.deleted
          });
        }
      }
      return nativeConnectors;
    });
    (0, _defineProperty2.default)(this, "getConnectorPackagePolicies", async () => {
      const kuery = `${_common.PACKAGE_POLICY_SAVED_OBJECT_TYPE}.package.name:${pkgName}`;
      this.logger.debug(`Fetching policies with kuery: "${kuery}"`);
      const policiesIterator = await this.packagePolicyService.fetchAllItems(this.soClient, {
        perPage: 50,
        kuery
      });
      const policiesMetadata = [];
      for await (const policyPage of policiesIterator) {
        for (const policy of policyPage) {
          for (const input of policy.inputs) {
            if (input.type === connectorsInputName) {
              if (input.compiled_input != null) {
                var _policy$package;
                if (input.compiled_input.service_type == null) {
                  this.logger.debug(`Policy ${policy.id} is missing service_type, skipping`);
                  continue;
                }
                if (input.compiled_input.connector_id == null) {
                  this.logger.debug(`Policy ${policy.id} is missing connector_id`);
                  // No need to skip, that's fine
                }
                if (input.compiled_input.connector_name == null) {
                  this.logger.debug(`Policy ${policy.id} is missing connector_name`);
                  // No need to skip, that's fine
                }

                // TODO: We manage all policies here, not only agentless.
                // Return this code back once this logic is ironed out
                // if (policy.supports_agentless !== true) {
                //   this.logger.debug(`Policy ${policy.id} does not support agentless, skipping`);
                //   continue;
                // }

                policiesMetadata.push({
                  package_policy_id: policy.id,
                  package_policy_name: policy.name,
                  package_name: ((_policy$package = policy.package) === null || _policy$package === void 0 ? void 0 : _policy$package.name) || '',
                  agent_policy_ids: policy.policy_ids,
                  connector_settings: {
                    id: input.compiled_input.connector_id,
                    name: input.compiled_input.connector_name || '',
                    service_type: input.compiled_input.service_type
                  }
                });
              }
            }
          }
        }
      }
      return policiesMetadata;
    });
    (0, _defineProperty2.default)(this, "deployConnector", async connector => {
      var _this$soClient$getCur;
      this.logger.info(`Connector ${connector.id} has no integration policy associated with it, creating`);
      if (connector.id == null || connector.id.trim().length === 0) {
        throw new Error(`Connector id is null or empty`);
      }
      if (connector.is_deleted) {
        throw new Error(`Connector ${connector.id} has been deleted`);
      }
      if (connector.service_type == null || connector.service_type.trim().length === 0) {
        throw new Error(`Connector ${connector.id} service_type is null or empty`);
      }
      if (_searchConnectors.NATIVE_CONNECTOR_DEFINITIONS[connector.service_type] == null) {
        throw new Error(`Connector ${connector.id} service_type is incompatible with agentless or unsupported`);
      }
      this.logger.debug(`Getting package version for connectors package ${pkgName}`);
      const pkgVersion = await this.getPackageVersion();
      this.logger.debug(`Latest package version for ${pkgName} is ${pkgVersion}`);
      const agentPolicyToCreate = {
        name: `Agentless policy for ${connector.service_type} connector: ${connector.id}`,
        description: `Automatically generated on ${new Date(Date.now()).toISOString()}`,
        global_data_tags: [{
          name: 'organization',
          value: 'elastic'
        }, {
          name: 'division',
          value: 'engineering'
        }, {
          name: 'team',
          value: 'search-extract-and-transform'
        }],
        namespace: 'default',
        monitoring_enabled: ['logs', 'metrics'],
        inactivity_timeout: 3600,
        is_protected: false,
        supports_agentless: true
      };
      const packagePolicyToCreate = {
        package: {
          title: pkgTitle,
          name: pkgName,
          version: pkgVersion
        },
        name: `${connector.service_type} connector ${connector.id}`,
        description: '',
        namespace: '',
        enabled: true,
        inputs: [{
          type: connectorsInputName,
          policy_template: connector.service_type,
          enabled: true,
          vars: {
            connector_id: {
              type: 'string',
              value: connector.id
            },
            connector_name: {
              type: 'string',
              value: connector.name
            },
            service_type: {
              type: 'string',
              value: connector.service_type
            }
          },
          streams: []
        }],
        supports_agentless: true
      };
      const agentPolicy = await this.agentPolicyService.createWithPackagePolicies({
        soClient: this.soClient,
        esClient: this.esClient,
        agentPolicy: agentPolicyToCreate,
        packagePolicies: [packagePolicyToCreate],
        options: {
          withSysMonitoring: true,
          spaceId: (_this$soClient$getCur = this.soClient.getCurrentNamespace()) !== null && _this$soClient$getCur !== void 0 ? _this$soClient$getCur : _common2.DEFAULT_SPACE_ID,
          forcePackagePolicyCreation: true
        }
      });
      this.logger.info(`Successfully created agent policy ${agentPolicy.id} for agentless connector ${connector.id}`);
      return agentPolicy;
    });
    (0, _defineProperty2.default)(this, "removeDeployment", async packagePolicyId => {
      this.logger.info(`Deleting package policy ${packagePolicyId}`);
      const policy = await this.packagePolicyService.get(this.soClient, packagePolicyId);
      if (policy == null) {
        throw new Error(`Failed to delete policy ${packagePolicyId}: not found`);
      }
      await this.packagePolicyService.delete(this.soClient, this.esClient, [policy.id]);
      this.logger.debug(`Deleting package policies with ids ${policy.policy_ids}`);

      // TODO: can we do it in one go?
      // Why not use deleteFleetServerPoliciesForPolicyId?
      for (const agentPolicyId of policy.policy_ids) {
        this.logger.info(`Deleting agent policy ${agentPolicyId}`);
        await this.agentPolicyService.delete(this.soClient, this.esClient, agentPolicyId);
      }
    });
    (0, _defineProperty2.default)(this, "getAgentPolicyForConnectorId", async ({
      connectorId
    }) => {
      const allPolicies = await this.getConnectorPackagePolicies();
      const policies = getPoliciesByConnectorId(allPolicies, connectorId);
      if (!policies || policies.length === 0) {
        return null;
      }
      const [policy] = policies;
      if (policy && policy.agent_policy_ids.length > 0) {
        const policyId = policy.agent_policy_ids[0];
        const listAgentsResponse = await this.agentService.asInternalUser.listAgents({
          kuery: `fleet-agents.policy_id:${policyId}`,
          showInactive: false
        });
        if (!listAgentsResponse || listAgentsResponse.agents.length === 0) {
          // If no agents assigned to policy, just return the policy
          return policy;
        } else {
          if (listAgentsResponse.agents.length > 1) {
            this.logger.warn(`More than one agent assigned to policy ${policyId} that manages connector with id ${connectorId}`);
          }

          // Return the first (and only) agentless host associated with this policy
          return {
            ...policy,
            agent_metadata: {
              id: listAgentsResponse.agents[0].id,
              last_checkin_status: listAgentsResponse.agents[0].last_checkin_status,
              status: listAgentsResponse.agents[0].status
            }
          };
        }
      }
      return policy;
    });
    (0, _defineProperty2.default)(this, "getPackageVersion", async () => {
      this.logger.debug(`Fetching ${pkgName} version`);

      // This will raise an error if package is not there.
      // Situation is exceptional, so we can just show
      // the error message from getPackageInfo in this case
      const packageInfo = await (0, _packages.getPackageInfo)({
        savedObjectsClient: this.soClient,
        pkgName,
        pkgVersion: '',
        skipArchive: true,
        ignoreUnverified: true,
        prerelease: true
      });
      this.logger.debug(`Found ${pkgName} version: ${packageInfo.version}`);
      return packageInfo.version;
    });
    this.logger = logger;
    this.soClient = soClient;
    this.esClient = esClient;
    this.packagePolicyService = packagePolicyService;
    this.agentService = agentService;
    this.agentPolicyService = agentPolicyService;
  }
}
exports.AgentlessConnectorsInfraService = AgentlessConnectorsInfraService;
const getConnectorsToDeploy = (packagePolicies, connectors) => {
  const results = [];
  for (const connector of connectors) {
    // Skip deleted connectors
    if (connector.is_deleted) continue;

    // If no package policies reference this connector by id then it should be deployed
    if (packagePolicies.every(packagePolicy => connector.id !== packagePolicy.connector_settings.id && connector.id !== packagePolicy.package_policy_id)) {
      results.push(connector);
    }
  }
  return results;
};
exports.getConnectorsToDeploy = getConnectorsToDeploy;
const getPoliciesToDelete = (packagePolicies, connectors) => {
  const results = [];
  for (const packagePolicy of packagePolicies) {
    // If there is a connector that has been soft-deleted for this package policy then this policy should be deleted
    if (connectors.some(connector => connector.id === packagePolicy.connector_settings.id && connector.is_deleted === true)) {
      results.push(packagePolicy);
    }
  }
  return results;
};
exports.getPoliciesToDelete = getPoliciesToDelete;
const getPoliciesByConnectorId = (packagePolicies, connectorId) => {
  return packagePolicies.filter(packagePolicy => packagePolicy.connector_settings.id === connectorId || packagePolicy.package_policy_id === connectorId);
};
exports.getPoliciesByConnectorId = getPoliciesByConnectorId;
const getConnectorPolicyId = (packagePolicies, connectorId) => {
  return getPoliciesByConnectorId(packagePolicies, connectorId).map(policy => policy.package_policy_id);
};
exports.getConnectorPolicyId = getConnectorPolicyId;