"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AnalyticsIndex = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _constants = require("./constants");
var _full_jitter_backoff = require("../common/retry_service/full_jitter_backoff");
var _backfill_task = require("./tasks/backfill_task");
var _retry_service = require("./retry_service");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class AnalyticsIndex {
  constructor({
    logger,
    esClient,
    isServerless,
    indexName,
    indexAlias,
    indexVersion,
    mappings,
    painlessScriptId,
    painlessScript,
    taskManager,
    taskId,
    sourceIndex,
    sourceQuery
  }) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "indexName", void 0);
    (0, _defineProperty2.default)(this, "indexAlias", void 0);
    (0, _defineProperty2.default)(this, "indexVersion", void 0);
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "mappings", void 0);
    (0, _defineProperty2.default)(this, "indexSettings", void 0);
    (0, _defineProperty2.default)(this, "painlessScriptId", void 0);
    (0, _defineProperty2.default)(this, "painlessScript", void 0);
    (0, _defineProperty2.default)(this, "retryService", void 0);
    (0, _defineProperty2.default)(this, "taskManager", void 0);
    (0, _defineProperty2.default)(this, "taskId", void 0);
    (0, _defineProperty2.default)(this, "sourceIndex", void 0);
    (0, _defineProperty2.default)(this, "sourceQuery", void 0);
    this.logger = logger;
    this.esClient = esClient;
    this.indexName = indexName;
    this.indexAlias = indexAlias;
    this.indexVersion = indexVersion;
    this.mappings = mappings;
    this.mappings._meta = this.getMappingMeta({
      indexVersion,
      painlessScriptId
    });
    this.painlessScriptId = painlessScriptId;
    this.painlessScript = painlessScript;
    this.taskManager = taskManager;
    this.taskId = taskId;
    this.sourceIndex = sourceIndex;
    this.sourceQuery = sourceQuery;
    this.indexSettings = {
      hidden: true,
      // settings are not supported on serverless ES
      ...(isServerless ? {} : {
        number_of_shards: _constants.CAI_NUMBER_OF_SHARDS,
        auto_expand_replicas: _constants.CAI_AUTO_EXPAND_REPLICAS,
        refresh_interval: _constants.CAI_REFRESH_INTERVAL,
        mode: _constants.CAI_INDEX_MODE
      })
    };
    /**
     * We should wait at least 5ms before retrying and no more that 2sec
     */
    const backOffFactory = (0, _full_jitter_backoff.fullJitterBackoffFactory)({
      baseDelay: 5,
      maxBackoffTime: 2000
    });
    this.retryService = new _retry_service.CasesAnalyticsRetryService(this.logger, backOffFactory);
  }
  async upsertIndex() {
    try {
      await this.retryService.retryWithBackoff(() => this._upsertIndex());
    } catch (error) {
      // We do not throw because errors should not break execution
      this.logger.error(`[${this.indexName}] Failed to create index. Error message: ${error.message}`);
    }
  }
  async _upsertIndex() {
    try {
      const indexExists = await this.indexExists();
      if (!indexExists) {
        this.logDebug('Index does not exist. Creating.');
        await this.createIndexMapping();
      } else {
        this.logDebug('Index exists. Checking mapping.');
        await this.updateIndexMapping();
      }
    } catch (error) {
      var _error$body, _error$body$error, _error$body2, _error$body2$error;
      if (((_error$body = error.body) === null || _error$body === void 0 ? void 0 : (_error$body$error = _error$body.error) === null || _error$body$error === void 0 ? void 0 : _error$body$error.type) === 'resource_already_exists_exception') {
        this.logDebug('Index already exists. Skipping creation.');
        return;
      }
      if (((_error$body2 = error.body) === null || _error$body2 === void 0 ? void 0 : (_error$body2$error = _error$body2.error) === null || _error$body2$error === void 0 ? void 0 : _error$body2$error.type) === 'multi_project_pending_exception') {
        this.logDebug('Multi-project setup. Skipping creation.');
        return;
      }
      this.handleError('Failed to create the index.', error);
    }
  }
  async updateIndexMapping() {
    try {
      const shouldUpdateMapping = await this.shouldUpdateMapping();
      if (shouldUpdateMapping) {
        await this.updateMapping();
      } else {
        this.logDebug('Mapping version is up to date. Skipping update.');
      }
    } catch (error) {
      this.handleError('Failed to update the index mapping.', error);
    }
  }
  async getCurrentMapping() {
    return this.esClient.indices.getMapping({
      index: this.indexName
    });
  }
  async updateMapping() {
    this.logDebug(`Updating the painless script.`);
    await this.putScript();
    this.logDebug(`Updating index mapping.`);
    await this.esClient.indices.putMapping({
      index: this.indexName,
      ...this.mappings
    });
    this.logDebug(`Scheduling the backfill task.`);
    await this.scheduleBackfillTask();
  }
  async createIndexMapping() {
    this.logDebug(`Creating painless script.`);
    await this.putScript();
    this.logDebug(`Creating index.`);
    await this.esClient.indices.create({
      index: this.indexName,
      timeout: _constants.CAI_DEFAULT_TIMEOUT,
      mappings: this.mappings,
      settings: {
        index: this.indexSettings
      },
      aliases: {
        [this.indexAlias]: {
          is_write_index: true
        }
      }
    });
    this.logDebug(`Scheduling the backfill task.`);
    await this.scheduleBackfillTask();
  }
  async indexExists() {
    this.logDebug(`Checking if index exists.`);
    return this.esClient.indices.exists({
      index: this.indexName
    });
  }
  async shouldUpdateMapping() {
    var _currentMapping$this$;
    const currentMapping = await this.getCurrentMapping();
    return ((_currentMapping$this$ = currentMapping[this.indexName].mappings._meta) === null || _currentMapping$this$ === void 0 ? void 0 : _currentMapping$this$.mapping_version) < this.indexVersion;
  }
  async putScript() {
    await this.esClient.putScript({
      id: this.painlessScriptId,
      script: this.painlessScript
    });
  }
  getMappingMeta({
    indexVersion,
    painlessScriptId
  }) {
    this.logDebug(`Construction mapping._meta. Index version: ${indexVersion}. Painless script: ${painlessScriptId}.`);
    return {
      mapping_version: indexVersion,
      painless_script_id: painlessScriptId
    };
  }
  logDebug(message) {
    this.logger.debug(`[${this.indexName}] ${message}`, {
      tags: ['cai-index-creation', this.indexName]
    });
  }
  handleError(message, error) {
    this.logger.error(`[${this.indexName}] ${message} Error message: ${error.message}`);
    throw error;
  }
  async scheduleBackfillTask() {
    await (0, _backfill_task.scheduleCAIBackfillTask)({
      taskId: this.taskId,
      sourceIndex: this.sourceIndex,
      sourceQuery: this.sourceQuery,
      destIndex: this.indexName,
      taskManager: this.taskManager,
      logger: this.logger
    });
  }
}
exports.AnalyticsIndex = AnalyticsIndex;