"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerAnalyzeLogsRoutes = registerAnalyzeLogsRoutes;
var _server = require("@kbn/data-plugin/server");
var _apm = require("@kbn/langchain/server/tracers/apm");
var _langsmith = require("@kbn/langchain/server/tracers/langsmith");
var _common = require("../../common");
var _constants = require("../constants");
var _graph = require("../graphs/log_type_detection/graph");
var _route_validation = require("../util/route_validation");
var _with_availability = require("./with_availability");
var _errors = require("../lib/errors");
var _routes_util = require("./routes_util");
var _constants2 = require("../../common/constants");
var _cef_error = require("../lib/errors/cef_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerAnalyzeLogsRoutes(router) {
  router.versioned.post({
    path: _common.ANALYZE_LOGS_PATH,
    access: 'internal',
    options: {
      timeout: {
        idleSocket: _constants.ROUTE_HANDLER_TIMEOUT
      }
    },
    security: {
      authz: {
        requiredPrivileges: [_constants.FLEET_ALL_ROLE, _constants.INTEGRATIONS_ALL_ROLE]
      }
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: (0, _route_validation.buildRouteValidationWithZod)(_common.AnalyzeLogsRequestBody)
      }
    }
  }, (0, _with_availability.withAvailability)(async (context, request, res) => {
    const {
      packageName,
      dataStreamName,
      packageTitle,
      dataStreamTitle,
      logSamples,
      langSmithOptions
    } = request.body;
    const services = await context.resolve(['core']);
    const {
      client
    } = services.core.elasticsearch;
    const {
      getStartServices,
      logger
    } = await context.automaticImport;
    const [, startPlugins] = await getStartServices();
    try {
      var _langSmithOptions$pro;
      const inference = await startPlugins.inference;
      const abortSignal = (0, _server.getRequestAbortedSignal)(request.events.aborted$);
      const connectorId = request.body.connectorId;
      const model = await inference.getChatModel({
        request,
        connectorId,
        chatModelOptions: {
          // not passing specific `model`, we'll always use the connector default model
          // temperature may need to be parametrized in the future
          temperature: 0.05,
          // Only retry once inside the model call, we already handle backoff retries in the task runner for the entire task
          maxRetries: 1,
          // Disable streaming explicitly
          disableStreaming: true,
          // Set a hard limit of 50 concurrent requests
          maxConcurrency: 50,
          telemetryMetadata: {
            pluginId: 'automatic_import'
          },
          signal: abortSignal
        }
      });
      const options = {
        callbacks: [new _apm.APMTracer({
          projectName: (_langSmithOptions$pro = langSmithOptions === null || langSmithOptions === void 0 ? void 0 : langSmithOptions.projectName) !== null && _langSmithOptions$pro !== void 0 ? _langSmithOptions$pro : 'default'
        }, logger), ...(0, _langsmith.getLangSmithTracer)({
          ...langSmithOptions,
          logger
        })]
      };
      const logFormatParameters = {
        packageName,
        dataStreamName,
        packageTitle,
        dataStreamTitle,
        logSamples
      };
      const graph = await (0, _graph.getLogFormatDetectionGraph)({
        model,
        client
      });
      const graphResults = await graph.withConfig({
        runName: 'Log Format'
      }).invoke(logFormatParameters, options);
      const graphLogFormat = graphResults.results.samplesFormat.name;
      switch (graphLogFormat) {
        case 'unsupported':
          throw new _errors.UnsupportedLogFormatError({
            message: _constants2.GenerationErrorCode.UNSUPPORTED_LOG_SAMPLES_FORMAT
          });
        case 'cef':
          throw new _cef_error.CefError(_constants2.GenerationErrorCode.CEF_ERROR);
        case 'leef':
          throw new _errors.UnsupportedLogFormatError({
            message: _constants2.GenerationErrorCode.UNSUPPORTED_LOG_SAMPLES_FORMAT,
            logFormat: 'Log Event Extended Format (LEEF)'
          });
        case 'fix':
          throw new _errors.UnsupportedLogFormatError({
            message: _constants2.GenerationErrorCode.UNSUPPORTED_LOG_SAMPLES_FORMAT,
            logFormat: 'Financial Information eXchange (FIX)'
          });
      }
      return res.ok({
        body: _common.AnalyzeLogsResponse.parse(graphResults)
      });
    } catch (err) {
      try {
        (0, _routes_util.handleCustomErrors)(err, _constants2.GenerationErrorCode.RECURSION_LIMIT_ANALYZE_LOGS);
      } catch (e) {
        if ((0, _errors.isErrorThatHandlesItsOwnResponse)(e)) {
          return e.sendResponse(res);
        }
      }
      return res.badRequest({
        body: err
      });
    }
  }));
}