"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getErrorsByDocId = getErrorsByDocId;
exports.getUnifiedTraceItems = getUnifiedTraceItems;
var _utils = require("@kbn/apm-data-access-plugin/server/utils");
var _common = require("@kbn/observability-plugin/common");
var _server = require("@kbn/observability-plugin/server");
var _apm = require("../../../common/es_fields/apm");
var _as_mutable_array = require("../../../common/utils/as_mutable_array");
var _get_trace_items = require("./get_trace_items");
var _get_unified_trace_errors = require("./get_unified_trace_errors");
var _parse_otel_duration = require("../../lib/helpers/parse_otel_duration");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const fields = (0, _as_mutable_array.asMutableArray)(['@timestamp', 'trace.id', 'service.name']);
const optionalFields = (0, _as_mutable_array.asMutableArray)([_apm.SPAN_ID, _apm.SPAN_NAME, _apm.DURATION, _apm.SPAN_DURATION, _apm.TRANSACTION_DURATION, _apm.TRANSACTION_ID, _apm.TRANSACTION_NAME, _apm.PROCESSOR_EVENT, _apm.PARENT_ID, _apm.STATUS_CODE, _apm.TIMESTAMP_US, _apm.EVENT_OUTCOME, _apm.STATUS_CODE, _apm.SPAN_TYPE, _apm.SPAN_SUBTYPE, _apm.KIND]);
function getErrorsByDocId(unifiedTraceErrors) {
  const groupedErrorsByDocId = {};
  function addError(id, errorDocId) {
    if (!groupedErrorsByDocId[id]) {
      groupedErrorsByDocId[id] = [];
    }
    groupedErrorsByDocId[id].push({
      errorDocId
    });
  }
  unifiedTraceErrors.apmErrors.forEach(errorDoc => {
    var _errorDoc$transaction, _errorDoc$span;
    const id = ((_errorDoc$transaction = errorDoc.transaction) === null || _errorDoc$transaction === void 0 ? void 0 : _errorDoc$transaction.id) || ((_errorDoc$span = errorDoc.span) === null || _errorDoc$span === void 0 ? void 0 : _errorDoc$span.id);
    if (id) {
      addError(id, errorDoc.id);
    }
  });
  unifiedTraceErrors.unprocessedOtelErrors.forEach(errorDoc => errorDoc.spanId ? addError(errorDoc.spanId, errorDoc.id) : undefined);
  return groupedErrorsByDocId;
}

/**
 * Returns both APM documents and unprocessed OTEL spans
 */
async function getUnifiedTraceItems({
  apmEventClient,
  logsClient,
  maxTraceItemsFromUrlParam,
  traceId,
  start,
  end,
  config
}) {
  const maxTraceItems = maxTraceItemsFromUrlParam !== null && maxTraceItemsFromUrlParam !== void 0 ? maxTraceItemsFromUrlParam : config.ui.maxTraceItems;
  const size = Math.min(maxTraceItems, _get_trace_items.MAX_ITEMS_PER_PAGE);
  const unifiedTraceErrorsPromise = (0, _get_unified_trace_errors.getUnifiedTraceErrors)({
    apmEventClient,
    logsClient,
    traceId,
    start,
    end
  });
  const unifiedTracePromise = apmEventClient.search('get_unified_trace_items', {
    apm: {
      events: [_common.ProcessorEvent.span, _common.ProcessorEvent.transaction]
    },
    track_total_hits: true,
    size,
    query: {
      bool: {
        must: [{
          bool: {
            filter: [...(0, _server.termQuery)(_apm.TRACE_ID, traceId), ...(0, _server.rangeQuery)(start, end)],
            should: {
              exists: {
                field: _apm.PARENT_ID
              }
            }
          }
        }],
        should: [{
          terms: {
            [_apm.PROCESSOR_EVENT]: [_common.ProcessorEvent.span, _common.ProcessorEvent.transaction]
          }
        }, {
          bool: {
            must_not: {
              exists: {
                field: _apm.PROCESSOR_EVENT
              }
            }
          }
        }],
        minimum_should_match: 1
      }
    },
    fields: [...fields, ...optionalFields],
    sort: [{
      _score: 'asc'
    }, {
      _script: {
        type: 'number',
        script: {
          lang: 'painless',
          source: `$('${_apm.TRANSACTION_DURATION}', $('${_apm.SPAN_DURATION}', $('${_apm.DURATION}', 0)))`
        },
        order: 'desc'
      }
    }, {
      [_apm.AT_TIMESTAMP]: 'asc'
    }, {
      _doc: 'asc'
    }]
  }, {
    skipProcessorEventFilter: true
  });
  const [unifiedTraceErrors, unifiedTraceItems] = await Promise.all([unifiedTraceErrorsPromise, unifiedTracePromise]);
  const errorsByDocId = getErrorsByDocId(unifiedTraceErrors);
  return {
    traceItems: unifiedTraceItems.hits.hits.map(hit => {
      var _event$span, _event$span$duration, _event$transaction, _event$transaction$du, _event$span2, _event$transaction2, _event$span$id, _event$span3, _event$transaction3, _event$timestamp$us, _event$timestamp, _event$span$name, _event$span4, _event$transaction4, _event$event, _event$status, _event$event2, _event$event3, _event$status2, _event$parent, _event$span5, _event$span6;
      const event = (0, _utils.unflattenKnownApmEventFields)(hit.fields, fields);
      const apmDuration = ((_event$span = event.span) === null || _event$span === void 0 ? void 0 : (_event$span$duration = _event$span.duration) === null || _event$span$duration === void 0 ? void 0 : _event$span$duration.us) || ((_event$transaction = event.transaction) === null || _event$transaction === void 0 ? void 0 : (_event$transaction$du = _event$transaction.duration) === null || _event$transaction$du === void 0 ? void 0 : _event$transaction$du.us);
      const id = ((_event$span2 = event.span) === null || _event$span2 === void 0 ? void 0 : _event$span2.id) || ((_event$transaction2 = event.transaction) === null || _event$transaction2 === void 0 ? void 0 : _event$transaction2.id);
      if (!id) {
        return undefined;
      }
      const docErrors = errorsByDocId[id] || [];
      return {
        id: (_event$span$id = (_event$span3 = event.span) === null || _event$span3 === void 0 ? void 0 : _event$span3.id) !== null && _event$span$id !== void 0 ? _event$span$id : (_event$transaction3 = event.transaction) === null || _event$transaction3 === void 0 ? void 0 : _event$transaction3.id,
        timestampUs: (_event$timestamp$us = (_event$timestamp = event.timestamp) === null || _event$timestamp === void 0 ? void 0 : _event$timestamp.us) !== null && _event$timestamp$us !== void 0 ? _event$timestamp$us : toMicroseconds(event[_apm.AT_TIMESTAMP]),
        name: (_event$span$name = (_event$span4 = event.span) === null || _event$span4 === void 0 ? void 0 : _event$span4.name) !== null && _event$span$name !== void 0 ? _event$span$name : (_event$transaction4 = event.transaction) === null || _event$transaction4 === void 0 ? void 0 : _event$transaction4.name,
        traceId: event.trace.id,
        duration: resolveDuration(apmDuration, event.duration),
        ...((((_event$event = event.event) === null || _event$event === void 0 ? void 0 : _event$event.outcome) || ((_event$status = event.status) === null || _event$status === void 0 ? void 0 : _event$status.code)) && {
          status: {
            fieldName: (_event$event2 = event.event) !== null && _event$event2 !== void 0 && _event$event2.outcome ? _apm.EVENT_OUTCOME : _apm.STATUS_CODE,
            value: ((_event$event3 = event.event) === null || _event$event3 === void 0 ? void 0 : _event$event3.outcome) || ((_event$status2 = event.status) === null || _event$status2 === void 0 ? void 0 : _event$status2.code)
          }
        }),
        errors: docErrors,
        parentId: (_event$parent = event.parent) === null || _event$parent === void 0 ? void 0 : _event$parent.id,
        serviceName: event.service.name,
        type: ((_event$span5 = event.span) === null || _event$span5 === void 0 ? void 0 : _event$span5.subtype) || ((_event$span6 = event.span) === null || _event$span6 === void 0 ? void 0 : _event$span6.type) || event.kind
      };
    }).filter(_ => _),
    unifiedTraceErrors
  };
}

/**
 * Resolve either an APM or OTEL duration and if OTEL, format the duration from nanoseconds to microseconds.
 */
const resolveDuration = (apmDuration, otelDuration) => apmDuration !== null && apmDuration !== void 0 ? apmDuration : (0, _parse_otel_duration.parseOtelDuration)(otelDuration);
const toMicroseconds = ts => new Date(ts).getTime() * 1000; // Convert ms to us