"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDocumentSources = getDocumentSources;
var _server = require("@kbn/observability-plugin/server");
var _rollup = require("../../../common/rollup");
var _transactions = require("../../lib/helpers/transactions");
var _document_type = require("../../../common/document_type");
var _document_type2 = require("../../lib/helpers/create_es_client/document_type");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const QUERY_INDEX = {
  DOCUMENT_TYPE: 0,
  DURATION_SUMMARY_NOT_SUPPORTED: 1
};
const getRequest = ({
  documentType,
  rollupInterval,
  filters
}) => {
  return {
    apm: {
      sources: [{
        documentType,
        rollupInterval
      }]
    },
    track_total_hits: 1,
    size: 0,
    terminate_after: 1,
    query: {
      bool: {
        filter: filters
      }
    }
  };
};
async function getDocumentSources({
  apmEventClient,
  start,
  end,
  kuery
}) {
  const documentTypesToCheck = [_document_type.ApmDocumentType.ServiceTransactionMetric, _document_type.ApmDocumentType.TransactionMetric];
  const documentTypesInfo = await getDocumentTypesInfo({
    apmEventClient,
    start,
    end,
    kuery,
    documentTypesToCheck
  });
  return [...documentTypesInfo, {
    documentType: _document_type.ApmDocumentType.TransactionEvent,
    rollupInterval: _rollup.RollupInterval.None,
    hasDocs: true,
    hasDurationSummaryField: false
  }];
}
const getDocumentTypesInfo = async ({
  apmEventClient,
  start,
  end,
  kuery,
  documentTypesToCheck
}) => {
  const getRequests = getDocumentTypeRequestsFn({
    start,
    end,
    kuery
  });
  const sourceRequests = documentTypesToCheck.flatMap(getRequests);
  const allSearches = sourceRequests.flatMap(({
    documentTypeQuery,
    durationSummaryNotSupportedQuery
  }) => [documentTypeQuery, durationSummaryNotSupportedQuery]).filter(request => request !== undefined);
  const allResponses = (await apmEventClient.msearch('get_document_availability', ...allSearches)).responses;
  const hasAnyLegacyDocuments = sourceRequests.some(({
    documentType,
    rollupInterval
  }, index) => isLegacyDocType(documentType, rollupInterval) && allResponses[index + QUERY_INDEX.DURATION_SUMMARY_NOT_SUPPORTED].hits.total.value > 0);
  return sourceRequests.map(({
    documentType,
    rollupInterval,
    ...queries
  }) => {
    const numberOfQueries = Object.values(queries).filter(Boolean).length;
    // allResponses is sorted by the order of the requests in sourceRequests
    const docTypeResponses = allResponses.splice(0, numberOfQueries);
    const hasDocs = docTypeResponses[QUERY_INDEX.DOCUMENT_TYPE].hits.total.value > 0;
    // can only use >=8.7 document types (ServiceTransactionMetrics or TransactionMetrics with 10m and 60m intervals)
    // if there are no legacy documents
    const canUseContinousRollupDocs = hasDocs && !hasAnyLegacyDocuments;
    return {
      documentType,
      rollupInterval,
      hasDocs: isLegacyDocType(documentType, rollupInterval) ? hasDocs : canUseContinousRollupDocs,
      // all >=8.7 document types with rollups support duration summary
      hasDurationSummaryField: canUseContinousRollupDocs
    };
  });
};
const getDocumentTypeRequestsFn = ({
  start,
  end,
  kuery
}) => documentType => {
  const currentRange = (0, _server.rangeQuery)(start, end);
  const kql = (0, _server.kqlQuery)(kuery);
  const rollupIntervals = (0, _document_type2.getConfigForDocumentType)(documentType).rollupIntervals;
  return rollupIntervals.map(rollupInterval => ({
    documentType,
    rollupInterval,
    documentTypeQuery: getRequest({
      documentType,
      rollupInterval,
      filters: [...kql, ...currentRange]
    }),
    ...(isLegacyDocType(documentType, rollupInterval) ? {
      durationSummaryNotSupportedQuery: getRequest({
        documentType,
        rollupInterval,
        filters: [...kql, ...currentRange, (0, _transactions.isDurationSummaryNotSupportedFilter)()]
      })
    } : undefined)
  }));
};
const isLegacyDocType = (documentType, rollupInterval) => {
  return documentType === _document_type.ApmDocumentType.TransactionMetric && rollupInterval === _rollup.RollupInterval.OneMinute;
};