"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerCreateRuntimeFieldRouteLegacy = exports.registerCreateRuntimeFieldRoute = exports.createRuntimeField = void 0;
var _configSchema = require("@kbn/config-schema");
var _handle_errors = require("../util/handle_errors");
var _schemas = require("../../../schemas");
var _constants = require("../../../constants");
var _response_formatter = require("./response_formatter");
var _schema = require("../../schema");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the "Elastic License
 * 2.0", the "GNU Affero General Public License v3.0 only", and the "Server Side
 * Public License v 1"; you may not use this file except in compliance with, at
 * your election, the "Elastic License 2.0", the "GNU Affero General Public
 * License v3.0 only", or the "Server Side Public License, v 1".
 */

const createRuntimeField = async ({
  dataViewsService,
  usageCollection,
  counterName,
  id,
  name,
  runtimeField
}) => {
  usageCollection === null || usageCollection === void 0 ? void 0 : usageCollection.incrementCounter({
    counterName
  });
  const dataView = await dataViewsService.getDataViewLazy(id);
  const firstNameSegment = name.split('.')[0];
  const fld = Object.keys((await dataView.getFields({
    fieldName: [name, firstNameSegment]
  })).getFieldMap());

  // check getRuntimeField to cover composite fields
  if (fld.length || dataView.getRuntimeField(name) || dataView.getRuntimeField(firstNameSegment)) {
    throw new Error(`Field [name = ${name}] already exists.`);
  }
  const createdRuntimeFields = await dataView.addRuntimeField(name, runtimeField);
  await dataViewsService.updateSavedObject(dataView);
  return {
    dataView,
    fields: createdRuntimeFields
  };
};
exports.createRuntimeField = createRuntimeField;
const runtimeCreateFieldRouteFactory = (path, serviceKey, description) => (router, getStartServices, usageCollection) => {
  router.versioned.post({
    path,
    access: 'public',
    description,
    security: {
      authz: {
        requiredPrivileges: ['indexPatterns:manage']
      }
    }
  }).addVersion({
    version: _constants.INITIAL_REST_VERSION,
    validate: {
      request: {
        params: _configSchema.schema.object({
          id: _configSchema.schema.string({
            minLength: 1,
            maxLength: 1_000
          })
        }),
        body: _configSchema.schema.object({
          name: _configSchema.schema.string({
            minLength: 1,
            maxLength: 1_000
          }),
          runtimeField: _schemas.runtimeFieldSchema
        })
      },
      response: {
        200: {
          body: _schema.runtimeResponseSchema
        }
      }
    }
  }, (0, _handle_errors.handleErrors)(async (ctx, req, res) => {
    const core = await ctx.core;
    const savedObjectsClient = core.savedObjects.client;
    const elasticsearchClient = core.elasticsearch.client.asCurrentUser;
    const [,, {
      dataViewsServiceFactory
    }] = await getStartServices();
    const dataViewsService = await dataViewsServiceFactory(savedObjectsClient, elasticsearchClient, req);
    const id = req.params.id;
    const {
      name,
      runtimeField
    } = req.body;
    const {
      dataView,
      fields
    } = await createRuntimeField({
      dataViewsService,
      usageCollection,
      counterName: `${req.route.method} ${path}`,
      id,
      name,
      runtimeField: runtimeField
    });
    const response = await (0, _response_formatter.responseFormatter)({
      serviceKey,
      dataView,
      fields
    });
    return res.ok(response);
  }));
};
const registerCreateRuntimeFieldRoute = exports.registerCreateRuntimeFieldRoute = runtimeCreateFieldRouteFactory(_constants.RUNTIME_FIELD_PATH, _constants.SERVICE_KEY, _constants.CREATE_RUNTIME_FIELD_DESCRIPTION);
const registerCreateRuntimeFieldRouteLegacy = exports.registerCreateRuntimeFieldRouteLegacy = runtimeCreateFieldRouteFactory(_constants.RUNTIME_FIELD_PATH_LEGACY, _constants.SERVICE_KEY_LEGACY);