"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.groupAttackDiscoveryAlerts = void 0;
var _elasticAssistantCommon = require("@kbn/elastic-assistant-common");
var _constants = require("../../../../common/constants");
var _schema = require("./schema");
var _constants2 = require("../constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const groupAttackDiscoveryAlerts = alerts => {
  /**
   * First we should validate that the alerts array schema complies with the attack discovery object.
   */
  const attackDiscoveryAlerts = _schema.AttackDiscoveryExpandedAlertsSchema.validate(alerts, {}, undefined, {
    stripUnknownKeys: true
  });
  if (attackDiscoveryAlerts.length > _constants2.MAX_OPEN_CASES) {
    throw new Error(`Circuit breaker: Attack discovery alerts grouping would create more than the maximum number of allowed cases ${_constants2.MAX_OPEN_CASES}.`);
  }

  /**
   * For each attack discovery alert we would like to create one separate case.
   */
  const groupedAlerts = attackDiscoveryAlerts.map(attackAlert => {
    var _attackDiscovery$repl;
    const alertsIndexPattern = attackAlert.kibana.alert.rule.parameters.alertsIndexPattern;
    const attackDiscoveryId = attackAlert._id;
    const attackDiscovery = attackAlert.kibana.alert.attack_discovery;
    const alertIds = attackDiscovery.alert_ids;
    const replacements = Array.isArray(attackDiscovery.replacements) ? (0, _elasticAssistantCommon.transformInternalReplacements)(attackDiscovery.replacements) : undefined;
    const originalAlertIds = (0, _elasticAssistantCommon.getOriginalAlertIds)({
      alertIds,
      replacements
    });
    const caseTitle = attackDiscovery.title.slice(0, _constants.MAX_TITLE_LENGTH);
    const caseComments = [(0, _elasticAssistantCommon.getAttackDiscoveryMarkdown)({
      attackDiscovery: {
        id: attackDiscoveryId,
        alertIds,
        detailsMarkdown: attackDiscovery.details_markdown,
        entitySummaryMarkdown: attackDiscovery.entity_summary_markdown,
        mitreAttackTactics: attackDiscovery.mitre_attack_tactics,
        summaryMarkdown: attackDiscovery.summary_markdown,
        title: caseTitle
      },
      replacements: (_attackDiscovery$repl = attackDiscovery.replacements) === null || _attackDiscovery$repl === void 0 ? void 0 : _attackDiscovery$repl.reduce((acc, r) => {
        acc[r.uuid] = r.value;
        return acc;
      }, {})
    })].slice(0, _constants.MAX_DOCS_PER_PAGE / 2);

    /**
     * Each attack discovery alert references a list of SIEM alerts that led to the attack.
     * These SIEM alerts will be added to the case.
     */
    return {
      alerts: originalAlertIds.map(siemAlertId => ({
        _id: siemAlertId,
        _index: alertsIndexPattern
      })),
      grouping: {
        attack_discovery: attackDiscoveryId
      },
      comments: caseComments,
      title: caseTitle
    };
  });
  return groupedAlerts;
};
exports.groupAttackDiscoveryAlerts = groupAttackDiscoveryAlerts;