"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.similar = void 0;
var _lodash = require("lodash");
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _constants = require("../../../common/constants");
var _api = require("../../../common/types/api");
var _runtime_types = require("../../common/runtime_types");
var _error = require("../../common/error");
var _utils = require("../../common/utils");
var _authorization = require("../../authorization");
var _utils2 = require("../utils");
var _utils3 = require("../../authorization/utils");
var _observable_types = require("../observable_types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getSimilarities = (a, b, availableObservableTypes) => {
  const stringify = observable => [observable.typeKey, observable.value].join(',');
  const setA = new Set(a.attributes.observables.map(stringify));
  const setB = new Set(b.attributes.observables.map(stringify));
  const intersectingObservables = (0, _lodash.intersection)([...setA], [...setB]);
  return intersectingObservables.map(item => {
    var _availableObservableT;
    const [typeKey, value] = item.split(',');
    return {
      typeKey,
      value,
      typeLabel: (_availableObservableT = availableObservableTypes.get(typeKey)) === null || _availableObservableT === void 0 ? void 0 : _availableObservableT.label
    };
  }).filter(observable => availableObservableTypes.has(observable.typeKey));
};

/**
 * Retrieves cases similar to a given Case
 */
const similar = async (caseId, params, clientArgs, casesClient) => {
  const {
    services: {
      caseService,
      licensingService
    },
    logger,
    authorization
  } = clientArgs;
  const hasPlatinumLicenseOrGreater = await licensingService.isAtLeastPlatinum();
  if (!hasPlatinumLicenseOrGreater) {
    throw _boom.default.forbidden('In order to use the similar cases feature, you must be subscribed to an Elastic Platinum license');
  }
  try {
    const paramArgs = (0, _runtime_types.decodeWithExcessOrThrow)(_api.SimilarCasesSearchRequestRt)(params);
    const retrievedCase = await caseService.getCase({
      id: caseId
    });
    const availableObservableTypesMap = await (0, _observable_types.getAvailableObservableTypesMap)(casesClient, retrievedCase.attributes.owner);
    const ownerFilter = (0, _utils2.buildFilter)({
      filters: retrievedCase.attributes.owner,
      field: _constants.OWNER_FIELD,
      operator: 'or'
    });
    const {
      filter: authorizationFilter,
      ensureSavedObjectsAreAuthorized
    } = await authorization.getAuthorizationFilter(_authorization.Operations.findCases);
    const similarCasesFilter = (0, _utils2.buildObservablesFieldsFilter)(retrievedCase.attributes.observables.reduce((observableMap, observable) => {
      // NOTE: skip non-existent observable types
      if (!availableObservableTypesMap.has(observable.typeKey)) {
        return observableMap;
      }
      if (!observableMap[observable.typeKey]) {
        observableMap[observable.typeKey] = [];
      }
      observableMap[observable.typeKey].push(observable.value);
      return observableMap;
    }, {}));

    // NOTE: empty similar cases filter means that we are unable to show similar cases
    // and should not combine it with general filters below.
    if (!similarCasesFilter) {
      var _paramArgs$perPage;
      return {
        cases: [],
        page: 1,
        per_page: (_paramArgs$perPage = paramArgs.perPage) !== null && _paramArgs$perPage !== void 0 ? _paramArgs$perPage : 0,
        total: 0
      };
    }
    const filters = (0, _utils2.combineFilters)([similarCasesFilter, ownerFilter]);
    const finalCasesFilter = (0, _utils3.combineFilterWithAuthorizationFilter)(filters, authorizationFilter);
    const cases = await caseService.findCases({
      filter: finalCasesFilter,
      sortField: _utils.defaultSortField,
      search: `-"cases:${caseId}"`,
      rootSearchFields: ['_id'],
      page: paramArgs.page,
      perPage: paramArgs.perPage
    });
    ensureSavedObjectsAreAuthorized(cases.saved_objects.map(caseSavedObject => ({
      id: caseSavedObject.id,
      owner: caseSavedObject.attributes.owner
    })));
    const res = {
      cases: cases.saved_objects.map(so => ({
        ...(0, _utils.flattenCaseSavedObject)({
          savedObject: so
        }),
        similarities: {
          observables: getSimilarities(retrievedCase, so, availableObservableTypesMap)
        }
      })),
      page: cases.page,
      per_page: cases.per_page,
      total: cases.total
    };
    return (0, _runtime_types.decodeOrThrow)(_api.CasesSimilarResponseRt)(res);
  } catch (error) {
    throw (0, _error.createCaseError)({
      message: `Failed to find cases: ${JSON.stringify(params)}: ${error}`,
      error,
      logger
    });
  }
};
exports.similar = similar;