"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.APMEventClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _common = require("@kbn/observability-plugin/common");
var _server = require("@kbn/observability-plugin/server");
var _lodash = require("lodash");
var _exclude_tiers_query = require("@kbn/observability-utils-common/es/queries/exclude_tiers_query");
var _utils = require("../../../../utils");
var _cancel_es_request_on_abort = require("../cancel_es_request_on_abort");
var _call_async_with_debug = require("../call_async_with_debug");
var _get_request_base = require("./get_request_base");
var _tier_filter = require("../../tier_filter");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class APMEventClient {
  constructor(config) {
    var _config$options$exclu;
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "debug", void 0);
    (0, _defineProperty2.default)(this, "request", void 0);
    (0, _defineProperty2.default)(this, "indices", void 0);
    /** @deprecated Use {@link excludedDataTiers} instead.
     * See https://www.elastic.co/guide/en/kibana/current/advanced-options.html **/
    (0, _defineProperty2.default)(this, "includeFrozen", void 0);
    (0, _defineProperty2.default)(this, "excludedDataTiers", void 0);
    (0, _defineProperty2.default)(this, "inspectableEsQueriesMap", void 0);
    this.esClient = config.esClient;
    this.debug = config.debug;
    this.request = config.request;
    this.indices = config.indices;
    this.includeFrozen = config.options.includeFrozen;
    this.excludedDataTiers = (_config$options$exclu = config.options.excludedDataTiers) !== null && _config$options$exclu !== void 0 ? _config$options$exclu : [];
    this.inspectableEsQueriesMap = config.options.inspectableEsQueriesMap;
  }
  callAsyncWithDebug({
    requestType,
    params,
    cb,
    operationName
  }) {
    return (0, _call_async_with_debug.callAsyncWithDebug)({
      isCalledWithInternalUser: false,
      debug: this.debug,
      request: this.request,
      operationName,
      requestParams: params,
      inspectableEsQueriesMap: this.inspectableEsQueriesMap,
      cb: () => {
        const controller = new AbortController();
        const promise = (0, _utils.withApmSpan)(operationName, () => {
          return (0, _cancel_es_request_on_abort.cancelEsRequestOnAbort)(cb({
            signal: controller.signal,
            meta: true
          }), this.request, controller);
        });
        return (0, _server.unwrapEsResponse)(promise);
      }
    });
  }
  async search(operationName, params, options) {
    const {
      index,
      filters
    } = (0, _get_request_base.getRequestBase)({
      apm: params.apm,
      indices: this.indices,
      skipProcessorEventFilter: options === null || options === void 0 ? void 0 : options.skipProcessorEventFilter
    });
    if (this.excludedDataTiers.length > 0) {
      filters.push(...(0, _exclude_tiers_query.excludeTiersQuery)(this.excludedDataTiers));
    }
    const searchParams = {
      ...(0, _lodash.omit)(params, 'apm', 'body'),
      index,
      query: {
        bool: {
          filter: filters,
          must: (0, _lodash.compact)([params.query])
        }
      },
      body: params.body,
      ...(this.includeFrozen ? {
        ignore_throttled: false
      } : {}),
      ignore_unavailable: true,
      preference: 'any',
      expand_wildcards: ['open', 'hidden']
    };
    return this.callAsyncWithDebug({
      cb: opts => this.esClient.search(searchParams, opts),
      operationName,
      params: searchParams,
      requestType: 'search'
    });
  }
  async logEventSearch(operationName, params) {
    // Reusing indices configured for errors since both events and errors are stored as logs.
    const index = (0, _get_request_base.processorEventsToIndex)([_common.ProcessorEvent.error], this.indices);
    const filter = this.excludedDataTiers.length > 0 ? (0, _exclude_tiers_query.excludeTiersQuery)(this.excludedDataTiers) : undefined;
    const searchParams = {
      ...(0, _lodash.omit)(params, 'body'),
      index,
      body: params.body,
      query: {
        bool: {
          filter,
          must: (0, _lodash.compact)([params.query])
        }
      },
      ...(this.includeFrozen ? {
        ignore_throttled: false
      } : {}),
      ignore_unavailable: true,
      preference: 'any',
      expand_wildcards: ['open', 'hidden']
    };
    return this.callAsyncWithDebug({
      cb: opts => this.esClient.search(searchParams, opts),
      operationName,
      params: searchParams,
      requestType: 'search'
    });
  }
  async msearch(operationName, ...allParams) {
    const searches = allParams.map(params => {
      const {
        index,
        filters
      } = (0, _get_request_base.getRequestBase)({
        apm: params.apm,
        indices: this.indices
      });
      if (this.excludedDataTiers.length > 0) {
        filters.push(...(0, _exclude_tiers_query.excludeTiersQuery)(this.excludedDataTiers));
      }
      const searchParams = [{
        index,
        preference: 'any',
        ...(this.includeFrozen ? {
          ignore_throttled: false
        } : {}),
        ignore_unavailable: true,
        expand_wildcards: ['open', 'hidden']
      }, {
        ...(0, _lodash.omit)(params, 'apm', 'body'),
        body: params.body,
        query: {
          bool: {
            filter: (0, _lodash.compact)([params.query, ...filters])
          }
        }
      }];
      return searchParams;
    }).flat();
    return this.callAsyncWithDebug({
      cb: opts => this.esClient.msearch({
        searches
      }, opts),
      operationName,
      params: searches,
      requestType: 'msearch'
    });
  }
  async eqlSearch(operationName, params) {
    const index = (0, _get_request_base.processorEventsToIndex)(params.apm.events, this.indices);
    const requestParams = {
      ...(0, _lodash.omit)(params, 'apm'),
      index
    };
    return this.callAsyncWithDebug({
      operationName,
      requestType: 'eql_search',
      params: requestParams,
      cb: opts => this.esClient.eql.search(requestParams, opts)
    });
  }
  async fieldCaps(operationName, params) {
    const index = (0, _get_request_base.processorEventsToIndex)(params.apm.events, this.indices);
    const requestParams = {
      ...(0, _lodash.omit)(params, 'apm'),
      index,
      index_filter: (0, _tier_filter.getDataTierFilterCombined)({
        filter: params.index_filter,
        excludedDataTiers: this.excludedDataTiers
      })
    };
    return this.callAsyncWithDebug({
      operationName,
      requestType: '_field_caps',
      params: requestParams,
      cb: opts => this.esClient.fieldCaps(requestParams, opts)
    });
  }
  async termsEnum(operationName, params) {
    const index = (0, _get_request_base.processorEventsToIndex)(params.apm.events, this.indices);
    const requestParams = {
      ...(0, _lodash.omit)(params, 'apm'),
      index: index.join(','),
      index_filter: (0, _tier_filter.getDataTierFilterCombined)({
        filter: params.index_filter,
        excludedDataTiers: this.excludedDataTiers
      })
    };
    return this.callAsyncWithDebug({
      operationName,
      requestType: '_terms_enum',
      params: requestParams,
      cb: opts => this.esClient.termsEnum(requestParams, opts)
    });
  }
  getIndicesFromProcessorEvent(processorEvent) {
    return (0, _get_request_base.processorEventsToIndex)([processorEvent], this.indices);
  }
}
exports.APMEventClient = APMEventClient;