/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */
package org.elasticsearch.xpack.core.ilm;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexRequest;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.client.internal.Client;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.LifecycleExecutionState;
import org.elasticsearch.cluster.metadata.ProjectMetadata;
import org.elasticsearch.common.Strings;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.index.IndexNotFoundException;

import java.util.function.BiFunction;

/**
 * Deletes the index identified by the index name supplier.
 */
public class CleanupGeneratedIndexStep extends AsyncRetryDuringSnapshotActionStep {

    public static final String NAME = "cleanup-generated-index";

    private static final Logger logger = LogManager.getLogger(CleanupGeneratedIndexStep.class);

    private final BiFunction<String, LifecycleExecutionState, String> targetIndexNameSupplier;

    public CleanupGeneratedIndexStep(
        StepKey key,
        StepKey nextStepKey,
        Client client,
        BiFunction<String, LifecycleExecutionState, String> targetIndexNameSupplier
    ) {
        super(key, nextStepKey, client);
        this.targetIndexNameSupplier = targetIndexNameSupplier;
    }

    @Override
    public boolean isRetryable() {
        return true;
    }

    @Override
    void performDuringNoSnapshot(IndexMetadata indexMetadata, ProjectMetadata currentProject, ActionListener<Void> listener) {
        // If the index was generated by a resize operation, and the source index does not exist anymore, we skip the deletion to avoid
        // data loss.
        final String generatedIndexSource = IndexMetadata.INDEX_RESIZE_SOURCE_NAME.get(indexMetadata.getSettings());
        if (Strings.hasText(generatedIndexSource)) {
            if (currentProject.index(generatedIndexSource) == null) {
                String policyName = indexMetadata.getLifecyclePolicyName();
                logger.warn(
                    "managed index [{}] as part of policy [{}] is a shrunk index and the source index [{}] does not exist "
                        + "anymore. will skip the [{}] step",
                    indexMetadata.getIndex().getName(),
                    policyName,
                    generatedIndexSource,
                    NAME
                );
                listener.onResponse(null);
                return;
            }
        }

        final String targetIndexName = targetIndexNameSupplier.apply(
            indexMetadata.getIndex().getName(),
            indexMetadata.getLifecycleExecutionState()
        );
        // If no index name was generated, there is nothing for us to delete, so we can move on
        if (Strings.hasText(targetIndexName) == false) {
            listener.onResponse(null);
            return;
        }
        getClient(currentProject.id()).admin()
            .indices()
            .delete(new DeleteIndexRequest(targetIndexName).masterNodeTimeout(TimeValue.MAX_VALUE), new ActionListener<>() {
                @Override
                public void onResponse(AcknowledgedResponse acknowledgedResponse) {
                    // even if not all nodes acked the delete request yet we can consider this operation as successful as
                    // we'll generate a new index name and attempt to create a new index with the newly generated name
                    listener.onResponse(null);
                }

                @Override
                public void onFailure(Exception e) {
                    if (e instanceof IndexNotFoundException) {
                        // we can move on if the index was deleted in the meantime
                        listener.onResponse(null);
                    } else {
                        listener.onFailure(e);
                    }
                }
            });
    }

    public BiFunction<String, LifecycleExecutionState, String> getTargetIndexNameSupplier() {
        return targetIndexNameSupplier;
    }
}
