"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DataStreamDetailsClient = void 0;
var _ioTsUtils = require("@kbn/io-ts-utils");
var _api_types = require("../../../common/api_types");
var _integration = require("../../../common/data_streams_stats/integration");
var _errors = require("../../../common/errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class DataStreamDetailsClient {
  constructor(http, telemetryClient) {
    this.http = http;
    this.telemetryClient = telemetryClient;
  }
  async getDataStreamSettings({
    dataStream
  }) {
    const response = await this.http.get(`/internal/dataset_quality/data_streams/${dataStream}/settings`).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to fetch data stream settings": ${error}`, error);
    });
    const dataStreamSettings = (0, _ioTsUtils.decodeOrThrow)(_api_types.getDataStreamsSettingsResponseRt, message => new _errors.DatasetQualityError(`Failed to decode data stream settings response: ${message}"`))(response);
    return dataStreamSettings;
  }
  async getDataStreamDetails({
    dataStream,
    start,
    end
  }) {
    const response = await this.http.get(`/internal/dataset_quality/data_streams/${dataStream}/details`, {
      query: {
        start,
        end
      }
    }).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to fetch data stream details": ${error}`, error);
    });
    const dataStreamDetails = (0, _ioTsUtils.decodeOrThrow)(_api_types.getDataStreamsDetailsResponseRt, message => new _errors.DatasetQualityError(`Failed to decode data stream details response: ${message}"`))(response);
    return dataStreamDetails;
  }
  async getFailedDocsDetails({
    dataStream,
    start,
    end
  }) {
    const response = await this.http.get(`/internal/dataset_quality/data_streams/${dataStream}/failed_docs`, {
      query: {
        start,
        end
      }
    }).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to fetch data stream failed docs details": ${error}`, error);
    });
    return (0, _ioTsUtils.decodeOrThrow)(_api_types.qualityIssueBaseRT, message => new _errors.DatasetQualityError(`Failed to decode data stream failed docs details response: ${message}"`))(response);
  }
  async getFailedDocsErrors({
    dataStream,
    start,
    end
  }) {
    const response = await this.http.get(`/internal/dataset_quality/data_streams/${dataStream}/failed_docs/errors`, {
      query: {
        start,
        end
      }
    }).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to fetch data stream failed docs details": ${error}`, error);
    });
    return (0, _ioTsUtils.decodeOrThrow)(_api_types.failedDocsErrorsRt, message => new _errors.DatasetQualityError(`Failed to decode data stream failed docs details response: ${message}"`))(response);
  }
  async getDataStreamDegradedFields({
    dataStream,
    start,
    end
  }) {
    const response = await this.http.get(`/internal/dataset_quality/data_streams/${dataStream}/degraded_fields`, {
      query: {
        start,
        end
      }
    }).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to fetch data stream degraded fields": ${error}`, error);
    });
    return (0, _ioTsUtils.decodeOrThrow)(_api_types.getDataStreamDegradedFieldsResponseRt, message => new _errors.DatasetQualityError(`Failed to decode data stream degraded fields response: ${message}"`))(response);
  }
  async getDataStreamDegradedFieldValues({
    dataStream,
    degradedField
  }) {
    const response = await this.http.get(`/internal/dataset_quality/data_streams/${dataStream}/degraded_field/${degradedField}/values`).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to fetch data stream degraded field Value": ${error}`, error);
    });
    return (0, _ioTsUtils.decodeOrThrow)(_api_types.degradedFieldValuesRt, message => new _errors.DatasetQualityError(`Failed to decode data stream degraded field values response: ${message}"`))(response);
  }
  async checkAndLoadIntegration({
    dataStream
  }) {
    const response = await this.http.get(`/internal/dataset_quality/data_streams/${dataStream}/integration/check`).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to check if data stream belongs to an integration": ${error}`, error);
    });
    const decodedResponse = (0, _ioTsUtils.decodeOrThrow)(_api_types.checkAndLoadIntegrationResponseRt, message => new _errors.DatasetQualityError(`Failed to decode integration check response: ${message}"`))(response);
    return {
      ...decodedResponse,
      integration: decodedResponse.isIntegration ? _integration.Integration.create(decodedResponse.integration) : undefined
    };
  }
  async getIntegrationDashboards({
    integration
  }) {
    const response = await this.http.get(`/internal/dataset_quality/integrations/${integration}/dashboards`).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to fetch integration dashboards": ${error}`, error);
    });
    const {
      dashboards
    } = (0, _ioTsUtils.decodeOrThrow)(_api_types.integrationDashboardsRT, message => new _errors.DatasetQualityError(`Failed to decode integration dashboards response: ${message}"`))(response);
    return dashboards;
  }
  async analyzeDegradedField({
    dataStream,
    degradedField,
    lastBackingIndex
  }) {
    const response = await this.http.get(`/internal/dataset_quality/data_streams/${dataStream}/degraded_field/${degradedField}/analyze`, {
      query: {
        lastBackingIndex
      }
    }).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to analyze degraded field: ${degradedField} for datastream: ${dataStream}`, error);
    });
    return (0, _ioTsUtils.decodeOrThrow)(_api_types.degradedFieldAnalysisRt, message => new _errors.DatasetQualityError(`Failed to decode the analysis response: ${message}`))(response);
  }
  async setNewFieldLimit({
    dataStream,
    newFieldLimit
  }) {
    const response = await this.http.put(`/internal/dataset_quality/data_streams/${dataStream}/update_field_limit`, {
      body: JSON.stringify({
        newFieldLimit
      })
    }).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to set new Limit: ${error.message}`, error);
    });
    const decodedResponse = (0, _ioTsUtils.decodeOrThrow)(_api_types.updateFieldLimitResponseRt, message => new _errors.DatasetQualityError(`Failed to decode setting of new limit response: ${message}"`))(response);
    return decodedResponse;
  }
  async rolloverDataStream({
    dataStream
  }) {
    const response = await this.http.post(`/internal/dataset_quality/data_streams/${dataStream}/rollover`).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to rollover datastream": ${error}`, error);
    });
    return (0, _ioTsUtils.decodeOrThrow)(_api_types.dataStreamRolloverResponseRt, message => new _errors.DatasetQualityError(`Failed to decode rollover response: ${message}"`))(response);
  }
  async updateFailureStore({
    dataStream,
    failureStoreEnabled,
    customRetentionPeriod
  }) {
    const response = await this.http.put(`/internal/dataset_quality/data_streams/${dataStream}/update_failure_store`, {
      body: JSON.stringify({
        failureStoreEnabled,
        customRetentionPeriod
      })
    }).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to update failure store": ${error}`, error);
    });
    this.telemetryClient.trackFailureStoreUpdated({
      data_stream_name: dataStream,
      failure_store_enabled: failureStoreEnabled,
      custom_retention_period: customRetentionPeriod
    });
    return (0, _ioTsUtils.decodeOrThrow)(_api_types.updateFailureStoreResponseRt, message => new _errors.DatasetQualityError(`Failed to decode update failure store response: ${message}"`))(response);
  }
  async getNonAggregatableDatasets(params) {
    const response = await this.http.get(`/internal/dataset_quality/data_streams/${params.dataStream}/non_aggregatable`, {
      query: {
        start: params.start,
        end: params.end
      }
    }).catch(error => {
      throw new _errors.DatasetQualityError(`Failed to fetch non-aggregatable datasets: ${error}`, error);
    });
    return (0, _ioTsUtils.decodeOrThrow)(_api_types.getNonAggregatableDatasetsRt, message => new _errors.DatasetQualityError(`Failed to decode non-aggregatable datasets response: ${message}`))(response);
  }
}
exports.DataStreamDetailsClient = DataStreamDetailsClient;