"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.positionHandlerCreators = exports.layerHandlerCreators = exports.groupHandlerCreators = exports.clipboardHandlerCreators = exports.basicHandlerCreators = exports.alignmentDistributionHandlerCreators = void 0;
var _lodash = require("lodash");
var _clipboard = require("./clipboard");
var _clone_subgraphs = require("./clone_subgraphs");
var _get_id = require("./get_id");
var _constants = require("../../common/lib/constants");
var _canvas_notify_service = require("../services/canvas_notify_service");
var _canvas_custom_element_service = require("../services/canvas_custom_element_service");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const extractId = node => node.id;
// handlers for clone, delete, and saving custom elements
const basicHandlerCreators = exports.basicHandlerCreators = {
  cloneNodes: ({
    insertNodes,
    pageId,
    selectToplevelNodes,
    selectedNodes
  }) => () => {
    const clonedNodes = selectedNodes && (0, _clone_subgraphs.cloneSubgraphs)(selectedNodes);
    if (clonedNodes) {
      insertNodes(clonedNodes, pageId);
      selectToplevelNodes(clonedNodes);
    }
  },
  deleteNodes: ({
    pageId,
    removeNodes,
    selectedNodes
  }) => () => {
    if (selectedNodes.length) {
      removeNodes(selectedNodes.map(extractId), pageId);
    }
  },
  createCustomElement: ({
    selectedNodes
  }) => (name = '', description = '', image = '') => {
    const notifyService = (0, _canvas_notify_service.getCanvasNotifyService)();
    const customElementService = (0, _canvas_custom_element_service.getCustomElementService)();
    if (selectedNodes.length) {
      const content = JSON.stringify({
        selectedNodes
      });
      const customElement = {
        id: (0, _get_id.getId)('custom-element'),
        name: (0, _lodash.camelCase)(name),
        displayName: name,
        help: description,
        image,
        content
      };
      customElementService.create(customElement).then(() => notifyService.success(`Custom element '${customElement.displayName || customElement.id}' was saved`, {
        'data-test-subj': 'canvasCustomElementCreate-success'
      })).catch(error => notifyService.warning(error, {
        title: `Custom element '${customElement.displayName || customElement.id}' was not saved`
      }));
    }
  }
};

// handlers for alignment and distribution
const alignmentDistributionHandlerCreators = exports.alignmentDistributionHandlerCreators = Object.assign({}, ...['alignLeft', 'alignCenter', 'alignRight', 'alignTop', 'alignMiddle', 'alignBottom', 'distributeHorizontally', 'distributeVertically'].map(event => ({
  [event]: ({
    commit
  }) => () => {
    commit('actionEvent', {
      event
    });
  }
})));

// handlers for group and ungroup
const groupHandlerCreators = exports.groupHandlerCreators = {
  groupNodes: ({
    commit
  }) => () => {
    commit('actionEvent', {
      event: 'group'
    });
  },
  ungroupNodes: ({
    commit
  }) => () => {
    commit('actionEvent', {
      event: 'ungroup'
    });
  }
};

// handlers for cut/copy/paste
const clipboardHandlerCreators = exports.clipboardHandlerCreators = {
  cutNodes: ({
    pageId,
    removeNodes,
    selectedNodes
  }) => () => {
    const notifyService = (0, _canvas_notify_service.getCanvasNotifyService)();
    if (selectedNodes.length) {
      (0, _clipboard.setClipboardData)({
        selectedNodes
      });
      removeNodes(selectedNodes.map(extractId), pageId);
      notifyService.success('Cut element to clipboard');
    }
  },
  copyNodes: ({
    selectedNodes
  }) => () => {
    const notifyService = (0, _canvas_notify_service.getCanvasNotifyService)();
    if (selectedNodes.length) {
      (0, _clipboard.setClipboardData)({
        selectedNodes
      });
      notifyService.success('Copied element to clipboard');
    }
  },
  pasteNodes: ({
    insertNodes,
    pageId,
    selectToplevelNodes
  }) => () => {
    const {
      selectedNodes = []
    } = JSON.parse((0, _clipboard.getClipboardData)()) || {};
    const clonedNodes = selectedNodes && (0, _clone_subgraphs.cloneSubgraphs)(selectedNodes);
    if (clonedNodes) {
      insertNodes(clonedNodes, pageId); // first clone and persist the new node(s)
      selectToplevelNodes(clonedNodes); // then select the cloned node(s)
    }
  }
};

// handlers for changing element layer position
// TODO: support relayering multiple elements
const layerHandlerCreators = exports.layerHandlerCreators = {
  bringToFront: ({
    elementLayer,
    pageId,
    selectedNodes
  }) => () => {
    if (selectedNodes.length === 1) {
      elementLayer(pageId, selectedNodes[0].id, Infinity);
    }
  },
  bringForward: ({
    elementLayer,
    pageId,
    selectedNodes
  }) => () => {
    if (selectedNodes.length === 1) {
      elementLayer(pageId, selectedNodes[0].id, 1);
    }
  },
  sendBackward: ({
    elementLayer,
    pageId,
    selectedNodes
  }) => () => {
    if (selectedNodes.length === 1) {
      elementLayer(pageId, selectedNodes[0].id, -1);
    }
  },
  sendToBack: ({
    elementLayer,
    pageId,
    selectedNodes
  }) => () => {
    if (selectedNodes.length === 1) {
      elementLayer(pageId, selectedNodes[0].id, -Infinity);
    }
  }
};

// handlers for shifting elements up, down, left, and right
const positionHandlerCreators = exports.positionHandlerCreators = {
  shiftUp: ({
    selectedNodes,
    setMultiplePositions
  }) => () => {
    setMultiplePositions(selectedNodes.map(element => {
      element.position.top -= _constants.ELEMENT_SHIFT_OFFSET;
      return element;
    }));
  },
  shiftDown: ({
    selectedNodes,
    setMultiplePositions
  }) => () => {
    setMultiplePositions(selectedNodes.map(element => {
      element.position.top += _constants.ELEMENT_SHIFT_OFFSET;
      return element;
    }));
  },
  shiftLeft: ({
    selectedNodes,
    setMultiplePositions
  }) => () => {
    setMultiplePositions(selectedNodes.map(element => {
      element.position.left -= _constants.ELEMENT_SHIFT_OFFSET;
      return element;
    }));
  },
  shiftRight: ({
    selectedNodes,
    setMultiplePositions
  }) => () => {
    setMultiplePositions(selectedNodes.map(element => {
      element.position.left += _constants.ELEMENT_SHIFT_OFFSET;
      return element;
    }));
  },
  nudgeUp: ({
    selectedNodes,
    setMultiplePositions
  }) => () => {
    setMultiplePositions(selectedNodes.map(element => {
      element.position.top -= _constants.ELEMENT_NUDGE_OFFSET;
      return element;
    }));
  },
  nudgeDown: ({
    selectedNodes,
    setMultiplePositions
  }) => () => {
    setMultiplePositions(selectedNodes.map(element => {
      element.position.top += _constants.ELEMENT_NUDGE_OFFSET;
      return element;
    }));
  },
  nudgeLeft: ({
    selectedNodes,
    setMultiplePositions
  }) => () => {
    setMultiplePositions(selectedNodes.map(element => {
      element.position.left -= _constants.ELEMENT_NUDGE_OFFSET;
      return element;
    }));
  },
  nudgeRight: ({
    selectedNodes,
    setMultiplePositions
  }) => () => {
    setMultiplePositions(selectedNodes.map(element => {
      element.position.left += _constants.ELEMENT_NUDGE_OFFSET;
      return element;
    }));
  }
};