"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useAIAssistChat = useAIAssistChat;
var _react = require("react");
var _swr = _interopRequireDefault(require("swr"));
var _uuid = require("uuid");
var _api = require("../utils/api");
var _types = require("../types");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getStreamedResponse = async (api, chatRequest, mutate, extraMetadataRef, messagesRef, abortControllerRef) => {
  var _chatRequest$options, _chatRequest$options2;
  const previousMessages = messagesRef.current;
  mutate(chatRequest.messages, false);
  const constructedMessagesPayload = chatRequest.messages.map(({
    role,
    content
  }) => ({
    role,
    content
  }));
  return await (0, _api.fetchApi)({
    api,
    messages: constructedMessagesPayload,
    body: {
      data: chatRequest.data,
      ...extraMetadataRef.current.body,
      ...((_chatRequest$options = chatRequest.options) === null || _chatRequest$options === void 0 ? void 0 : _chatRequest$options.body)
    },
    headers: {
      ...extraMetadataRef.current.headers,
      ...((_chatRequest$options2 = chatRequest.options) === null || _chatRequest$options2 === void 0 ? void 0 : _chatRequest$options2.headers)
    },
    abortController: () => abortControllerRef.current,
    appendMessage(message) {
      mutate([...chatRequest.messages, message], false);
    },
    handleFailure(errorMessage) {
      const systemErrorMessage = {
        id: (0, _uuid.v4)(),
        content: errorMessage,
        role: _types.MessageRole.system,
        createdAt: new Date()
      };
      mutate([...previousMessages, chatRequest.messages.slice(-1)[0], systemErrorMessage], false);
    },
    onUpdate(merged) {
      mutate([...chatRequest.messages, ...merged], false);
    }
  });
};
function useAIAssistChat({
  api = '/api/chat',
  id = 'chat',
  initialInput = '',
  onError,
  headers,
  body
} = {}) {
  const chatKey = [typeof api === 'string' ? api : 'function', id];
  const [initialMessagesFallback] = (0, _react.useState)([]);

  // Store the chat state in SWR, using the chatId as the key to share states.
  const {
    data: messages,
    mutate
  } = (0, _swr.default)([chatKey, 'messages'], null, {
    fallbackData: initialMessagesFallback
  });
  const {
    data: isLoading = false,
    mutate: mutateLoading
  } = (0, _swr.default)([chatKey, 'loading'], null);
  const {
    data: error = undefined,
    mutate: setError
  } = (0, _swr.default)([chatKey, 'error'], null);
  const messagesRef = (0, _react.useRef)(messages || []);
  (0, _react.useEffect)(() => {
    messagesRef.current = messages || [];
  }, [messages]);
  const abortControllerRef = (0, _react.useRef)(null);
  const extraMetadataRef = (0, _react.useRef)({
    headers,
    body
  });
  (0, _react.useEffect)(() => {
    extraMetadataRef.current = {
      headers,
      body
    };
  }, [headers, body]);
  const triggerRequest = (0, _react.useCallback)(async chatRequest => {
    try {
      mutateLoading(true);
      setError(undefined);
      const abortController = new AbortController();
      abortControllerRef.current = abortController;
      await getStreamedResponse(api, chatRequest, mutate, extraMetadataRef, messagesRef, abortControllerRef);
      abortControllerRef.current = null;
    } catch (err) {
      if (err.name === 'AbortError') {
        abortControllerRef.current = null;
        return null;
      }
      if (onError && err instanceof Error) {
        onError(err);
      }
      setError(err);
    } finally {
      mutateLoading(false);
    }
  }, [mutate, mutateLoading, api, extraMetadataRef, onError, setError, messagesRef, abortControllerRef]);
  const append = (0, _react.useCallback)(async (message, {
    options,
    data
  } = {}) => {
    if (!message.id) {
      message.id = (0, _uuid.v4)();
    }
    const chatRequest = {
      messages: messagesRef.current.concat(message),
      options,
      data
    };
    return triggerRequest(chatRequest);
  }, [triggerRequest]);
  const reload = (0, _react.useCallback)(async ({
    options,
    data
  } = {}) => {
    if (messagesRef.current.length === 0) return null;
    const chatRequest = {
      messages: messagesRef.current.slice(0, messagesRef.current.length - 1),
      options,
      data
    };
    return triggerRequest(chatRequest);
  }, [triggerRequest]);
  const stop = (0, _react.useCallback)(() => {
    if (abortControllerRef.current) {
      abortControllerRef.current.abort();
      abortControllerRef.current = null;
    }
  }, []);
  const setMessages = (0, _react.useCallback)(newMessages => {
    mutate(newMessages, false);
    messagesRef.current = newMessages;
  }, [mutate]);
  const [input, setInput] = (0, _react.useState)(initialInput);
  const handleSubmit = (0, _react.useCallback)((e, options = {}, metadata) => {
    if (metadata) {
      extraMetadataRef.current = {
        ...extraMetadataRef.current,
        ...metadata
      };
    }
    e.preventDefault();
    if (!input) return;
    append({
      content: input,
      role: _types.MessageRole.user,
      createdAt: new Date()
    }, options);
    setInput('');
  }, [input, append]);
  const handleInputChange = e => {
    setInput(e.target.value);
  };
  return {
    messages: messages || [],
    error,
    append,
    reload,
    stop,
    setMessages,
    input,
    setInput,
    handleInputChange,
    handleSubmit,
    isLoading
  };
}