"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TagsCache = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Reactive client-side cache of the existing tags, connected to the TagsClient.
 *
 * Used (mostly) by the UI components to avoid performing http calls every time a component
 * needs to retrieve the list of all the existing tags or the tags associated with an object.
 */
class TagsCache {
  constructor({
    refreshHandler,
    refreshInterval
  }) {
    (0, _defineProperty2.default)(this, "refreshInterval", void 0);
    (0, _defineProperty2.default)(this, "refreshHandler", void 0);
    (0, _defineProperty2.default)(this, "intervalId", void 0);
    (0, _defineProperty2.default)(this, "internal$", void 0);
    (0, _defineProperty2.default)(this, "public$", void 0);
    (0, _defineProperty2.default)(this, "stop$", void 0);
    (0, _defineProperty2.default)(this, "isInitialized$", void 0);
    this.refreshHandler = refreshHandler;
    this.refreshInterval = refreshInterval;
    this.stop$ = new _rxjs.Subject();
    this.internal$ = new _rxjs.BehaviorSubject([]);
    this.public$ = this.internal$.pipe((0, _rxjs.takeUntil)(this.stop$));
    this.isInitialized$ = new _rxjs.BehaviorSubject(false);
  }
  async initialize() {
    await this.refresh();
    this.isInitialized$.next(true);
    if (this.refreshInterval) {
      this.intervalId = window.setInterval(() => {
        this.refresh();
      }, this.refreshInterval.asMilliseconds());
    }
  }
  async refresh() {
    try {
      const tags = await this.refreshHandler();
      this.internal$.next(tags);
    } catch (e) {
      // what should we do here?
    }
  }
  getState() {
    return this.internal$.getValue();
  }
  getState$({
    waitForInitialization = false
  } = {}) {
    return waitForInitialization ? this.isInitialized$.pipe((0, _rxjs.first)(isInitialized => isInitialized), (0, _rxjs.mergeMap)(() => this.public$)) : this.public$;
  }
  onDelete(id) {
    this.internal$.next(this.internal$.value.filter(tag => tag.id !== id));
  }
  onCreate(tag) {
    this.internal$.next([...this.internal$.value.filter(f => f.id !== tag.id), tag]);
  }
  onUpdate(id, attributes) {
    this.internal$.next(this.internal$.value.map(tag => {
      if (tag.id === id) {
        return {
          ...tag,
          ...attributes
        };
      }
      return tag;
    }));
  }
  onGetAll(tags) {
    this.internal$.next(tags);
  }
  stop() {
    if (this.intervalId) {
      window.clearInterval(this.intervalId);
    }
    this.stop$.next();
  }
}
exports.TagsCache = TagsCache;