"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerReportingUsageCollector = registerReportingUsageCollector;
var _rxjs = require("rxjs");
var _lodash = require("lodash");
var _collection_schema = require("./collection_schema");
var _task = require("./task");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getLatestTaskState(taskManager) {
  try {
    const result = await taskManager.get(_task.TASK_ID);
    return result;
  } catch (err) {
    const errMessage = err && err.message ? err.message : err.toString();
    /*
      The usage service WILL to try to fetch from this collector before the task manager has been initialized, because the
      task manager has to wait for all plugins to initialize first. It's fine to ignore it as next time around it will be
      initialized (or it will throw a different type of error)
    */
    if (!errMessage.includes('NotInitialized')) {
      throw err;
    }
  }
  return null;
}
function registerReportingUsageCollector(reporting, taskManager, usageCollection) {
  const isReportingReady = reporting.pluginStartsUp.bind(reporting);
  const getLicense = async () => {
    const {
      licensing
    } = await reporting.getPluginStartDeps();
    return await (0, _rxjs.firstValueFrom)(licensing.license$.pipe((0, _rxjs.map)(({
      isAvailable
    }) => ({
      isAvailable
    }))));
  };
  const collector = usageCollection.makeUsageCollector({
    type: 'reporting',
    fetch: async () => {
      const license = await getLicense();
      try {
        const doc = await getLatestTaskState(await taskManager);
        // get the accumulated state from the recurring task
        const {
          runs,
          ...state
        } = (0, _lodash.get)(doc, 'state');
        return {
          available: license.isAvailable === true,
          // is available under all non-expired licenses
          enabled: true,
          // is enabled, by nature of this code path executing
          ...state
        };
      } catch (err) {
        const errMessage = err && err.message ? err.message : err.toString();
        return {
          available: license.isAvailable === true,
          // is available under all non-expired licenses
          enabled: true,
          // is enabled, by nature of this code path executing
          has_errors: true,
          error_messages: [errMessage],
          number_of_scheduled_reports: 0,
          number_of_enabled_scheduled_reports: 0,
          number_of_scheduled_reports_by_type: {},
          number_of_enabled_scheduled_reports_by_type: {},
          number_of_scheduled_reports_with_notifications: 0
        };
      }
    },
    isReady: async () => {
      await taskManager;
      return isReportingReady();
    },
    schema: _collection_schema.ReportingSchema
  });
  usageCollection.registerCollector(collector);
}