"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IlmPolicyManager = void 0;
var _reportingCommon = require("@kbn/reporting-common");
var _reportingServer = require("@kbn/reporting-server");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Responsible for detecting and provisioning the reporting ILM policy in stateful deployments.
 *
 * Uses the provided {@link ElasticsearchClient} to scope request privileges.
 */
class IlmPolicyManager {
  constructor(client) {
    this.client = client;
  }
  static create(opts) {
    return new IlmPolicyManager(opts.client);
  }

  /**
   * Check that the ILM policy exists
   */
  async doesIlmPolicyExist() {
    try {
      await this.client.ilm.getLifecycle({
        name: _reportingCommon.ILM_POLICY_NAME
      });
      return true;
    } catch (e) {
      if (e.statusCode === 404) {
        return false;
      }
      throw e;
    }
  }

  /**
   * This method is automatically called on the Stack Management > Reporting page, by the `` API for users with
   * privilege to manage ILM, to notify them when attention is needed to update the policy for any reason.
   */
  async checkIlmMigrationStatus() {
    if (!(await this.doesIlmPolicyExist())) {
      return 'policy-not-found';
    }
    const [reportingDataStreamSettings, reportingLegacyIndexSettings] = await Promise.all([this.client.indices.getSettings({
      index: _reportingServer.REPORTING_DATA_STREAM_WILDCARD
    }), this.client.indices.getSettings({
      index: _reportingServer.REPORTING_LEGACY_INDICES
    })]);
    const hasUnmanaged = settings => {
      var _settings$settings, _settings$settings$in, _settings$settings$in2, _settings$settings2, _settings$settings2$i;
      return (settings === null || settings === void 0 ? void 0 : (_settings$settings = settings.settings) === null || _settings$settings === void 0 ? void 0 : (_settings$settings$in = _settings$settings.index) === null || _settings$settings$in === void 0 ? void 0 : (_settings$settings$in2 = _settings$settings$in.lifecycle) === null || _settings$settings$in2 === void 0 ? void 0 : _settings$settings$in2.name) !== _reportingCommon.ILM_POLICY_NAME && (settings === null || settings === void 0 ? void 0 : (_settings$settings2 = settings.settings) === null || _settings$settings2 === void 0 ? void 0 : (_settings$settings2$i = _settings$settings2['index.lifecycle']) === null || _settings$settings2$i === void 0 ? void 0 : _settings$settings2$i.name) !== _reportingCommon.ILM_POLICY_NAME;
    };
    const hasUnmanagedDataStream = Object.values(reportingDataStreamSettings).some(hasUnmanaged);
    const hasUnmanagedIndices = Object.values(reportingLegacyIndexSettings).some(hasUnmanaged);
    return hasUnmanagedDataStream || hasUnmanagedIndices ? 'indices-not-managed-by-policy' : 'ok';
  }

  /**
   * Create the Reporting ILM policy
   */
  async createIlmPolicy() {
    await this.client.ilm.putLifecycle({
      name: _reportingCommon.ILM_POLICY_NAME,
      policy: {
        phases: {
          hot: {
            actions: {}
          }
        }
      }
    });
  }

  /**
   * Update the Data Stream index template with a link to the Reporting ILM policy
   */
  async linkIlmPolicy() {
    const putTemplateAcknowledged = await this.client.cluster.putComponentTemplate({
      name: _reportingServer.REPORTING_DATA_STREAM_COMPONENT_TEMPLATE,
      template: {
        settings: {
          lifecycle: {
            name: _reportingCommon.ILM_POLICY_NAME
          }
        }
      },
      create: false
    });
    let backingIndicesAcknowledged = {
      acknowledged: null
    };
    const backingIndicesExist = await this.client.indices.exists({
      index: _reportingServer.REPORTING_DATA_STREAM_ALIAS,
      expand_wildcards: ['hidden']
    });
    if (backingIndicesExist) {
      backingIndicesAcknowledged = await this.client.indices.putSettings({
        index: _reportingServer.REPORTING_DATA_STREAM_ALIAS,
        settings: {
          lifecycle: {
            name: _reportingCommon.ILM_POLICY_NAME
          }
        }
      });
    }
    return {
      putTemplateResponse: putTemplateAcknowledged,
      backingIndicesAcknowledged
    };
  }

  /**
   * Update datastream to use ILM policy. If legacy indices exist, this attempts to link
   * the ILM policy to them as well.
   */
  async migrateIndicesToIlmPolicy() {
    const {
      putTemplateResponse: {
        acknowledged: putTemplateAcknowledged
      },
      backingIndicesAcknowledged: {
        acknowledged: backingIndicesAcknowledged
      }
    } = await this.linkIlmPolicy();
    let legacyAcknowledged = null;
    const legacyExists = await this.client.indices.exists({
      index: _reportingServer.REPORTING_LEGACY_INDICES,
      expand_wildcards: ['hidden']
    });
    if (legacyExists) {
      const {
        acknowledged
      } = await this.client.indices.putSettings({
        index: _reportingServer.REPORTING_LEGACY_INDICES,
        settings: {
          lifecycle: {
            name: _reportingCommon.ILM_POLICY_NAME
          }
        }
      });
      legacyAcknowledged = acknowledged;
    }
    return {
      putTemplateAcknowledged,
      backingIndicesAcknowledged,
      legacyAcknowledged
    };
  }
}
exports.IlmPolicyManager = IlmPolicyManager;