"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.clearLayerProp = clearLayerProp;
exports.findLayerById = findLayerById;
exports.getLayerIndex = getLayerIndex;
exports.removeTrackedLayerState = removeTrackedLayerState;
exports.rollbackTrackedLayerState = rollbackTrackedLayerState;
exports.setLayer = setLayer;
exports.trackCurrentLayerState = trackCurrentLayerState;
exports.updateLayerInList = updateLayerInList;
exports.updateLayerSourceDescriptorProp = updateLayerSourceDescriptorProp;
var _copy_persistent_state = require("../copy_persistent_state");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function getLayerIndex(list, layerId) {
  return list.findIndex(({
    id
  }) => layerId === id);
}
function findLayerById(state, layerId) {
  return state.layerList.find(({
    id
  }) => layerId === id);
}
function clearLayerProp(state, layerId, propName) {
  if (!layerId) {
    return state;
  }
  const {
    layerList
  } = state;
  const layerIdx = getLayerIndex(layerList, layerId);
  if (layerIdx === -1) {
    return state;
  }
  const updatedLayer = {
    ...layerList[layerIdx]
  };
  delete updatedLayer[propName];
  const updatedList = [...layerList.slice(0, layerIdx), updatedLayer, ...layerList.slice(layerIdx + 1)];
  return {
    ...state,
    layerList: updatedList
  };
}
function updateLayerInList(state, layerId, attribute, newValue) {
  if (!layerId) {
    return state;
  }
  const {
    layerList
  } = state;
  const layerIdx = getLayerIndex(layerList, layerId);
  if (layerIdx === -1) {
    return state;
  }
  const updatedLayer = {
    ...layerList[layerIdx],
    [attribute]: newValue
  };
  const updatedList = [...layerList.slice(0, layerIdx), updatedLayer, ...layerList.slice(layerIdx + 1)];
  return {
    ...state,
    layerList: updatedList
  };
}
function updateLayerSourceDescriptorProp(state, layerId, propName, value) {
  const {
    layerList
  } = state;
  const layerIdx = getLayerIndex(layerList, layerId);
  const updatedLayer = {
    ...layerList[layerIdx],
    sourceDescriptor: {
      ...layerList[layerIdx].sourceDescriptor,
      [propName]: value
    }
  };
  const updatedList = [...layerList.slice(0, layerIdx), updatedLayer, ...layerList.slice(layerIdx + 1)];
  return {
    ...state,
    layerList: updatedList
  };
}
function trackCurrentLayerState(state, layerId) {
  const layer = findLayerById(state, layerId);
  const layerCopy = (0, _copy_persistent_state.copyPersistentState)(layer);
  return updateLayerInList(state, layerId, _copy_persistent_state.TRACKED_LAYER_DESCRIPTOR, layerCopy);
}
function removeTrackedLayerState(state, layerId) {
  const layer = findLayerById(state, layerId);
  if (!layer) {
    return state;
  }
  const copyLayer = {
    ...layer
  };
  delete copyLayer[_copy_persistent_state.TRACKED_LAYER_DESCRIPTOR];
  return {
    ...state,
    layerList: setLayer(state.layerList, copyLayer)
  };
}
function rollbackTrackedLayerState(state, layerId) {
  const layer = findLayerById(state, layerId);
  if (!layer) {
    return state;
  }
  const trackedLayerDescriptor = layer[_copy_persistent_state.TRACKED_LAYER_DESCRIPTOR];

  // this assumes that any nested temp-state in the layer-descriptor (e.g. of styles), is not relevant and can be recovered easily (e.g. this is not the case for __dataRequests)
  // That assumption is true in the context of this app, but not generalizable.
  // consider rewriting copyPersistentState to only strip the first level of temp state.
  const rolledbackLayer = {
    ...layer,
    ...trackedLayerDescriptor
  };
  delete rolledbackLayer[_copy_persistent_state.TRACKED_LAYER_DESCRIPTOR];
  return {
    ...state,
    layerList: setLayer(state.layerList, rolledbackLayer)
  };
}
function setLayer(layerList, layerDescriptor) {
  const layerIndex = getLayerIndex(layerList, layerDescriptor.id);
  if (layerIndex === -1) {
    return layerList;
  }
  const newLayerList = [...layerList];
  newLayerList[layerIndex] = layerDescriptor;
  return newLayerList;
}