"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getProcessListChart = void 0;
var _common = require("@kbn/metrics-data-access-plugin/common");
var _constants = require("@kbn/metrics-data-access-plugin/common/constants");
var _constants2 = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TS_COLUMNS = [{
  name: 'timestamp',
  type: 'date'
}, {
  name: 'metric_0',
  type: 'number'
}];
const getEcsProcessListChart = async (infraMetricsClient, {
  hostTerm,
  to,
  command
}) => {
  const from = to - 60 * 15 * 1000; // 15 minutes

  const response = await infraMetricsClient.search({
    track_total_hits: false,
    size: 0,
    query: {
      bool: {
        filter: [{
          range: {
            [_constants2.TIMESTAMP_FIELD]: {
              gte: from,
              lte: to,
              format: 'epoch_millis'
            }
          }
        }, {
          term: hostTerm
        }]
      }
    },
    aggs: {
      process: {
        filter: {
          bool: {
            must: [{
              match: {
                [_constants2.PROCESS_COMMANDLINE_FIELD]: command
              }
            }]
          }
        },
        aggs: {
          filteredProc: {
            terms: {
              field: _constants2.PROCESS_COMMANDLINE_FIELD,
              size: 1
            },
            aggs: {
              timeseries: {
                date_histogram: {
                  field: _constants2.TIMESTAMP_FIELD,
                  fixed_interval: '1m',
                  extended_bounds: {
                    min: from,
                    max: to
                  }
                },
                aggs: {
                  cpu: {
                    avg: {
                      field: 'system.process.cpu.total.pct'
                    }
                  },
                  memory: {
                    avg: {
                      field: 'system.process.memory.rss.pct'
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
  });
  const {
    buckets
  } = response.aggregations.process.filteredProc;
  const initialResponse = {
    cpu: {
      id: 'cpu',
      columns: TS_COLUMNS,
      rows: []
    },
    memory: {
      id: 'memory',
      columns: TS_COLUMNS,
      rows: []
    }
  };
  const timeseries = buckets.length > 0 ? buckets[0].timeseries.buckets.reduce((tsResult, tsBucket) => {
    tsResult.cpu.rows.push({
      metric_0: tsBucket.cpu.value,
      timestamp: tsBucket.key
    });
    tsResult.memory.rows.push({
      metric_0: tsBucket.memory.value !== null ? tsBucket.memory.value / 100 : null,
      // convert to ratio (0-1)
      timestamp: tsBucket.key
    });
    return tsResult;
  }, initialResponse) : initialResponse;
  return timeseries;
};
const getSemConvProcessListChart = async (infraMetricsClient, {
  hostTerm,
  to,
  command
}) => {
  const from = to - 60 * 15 * 1000; // 15 minutes

  const response = await infraMetricsClient.search({
    track_total_hits: false,
    size: 0,
    query: {
      bool: {
        filter: [{
          range: {
            [_constants2.TIMESTAMP_FIELD]: {
              gte: from,
              lte: to,
              format: 'epoch_millis'
            }
          }
        }, {
          term: hostTerm
        }, {
          term: {
            [_constants.EVENT_DATASET]: _common.HOST_METRICS_RECEIVER_OTEL
          }
        }]
      }
    },
    aggs: {
      process: {
        filter: {
          bool: {
            must: [{
              match: {
                [_constants2.PROCESS_COMMANDLINE_FIELD]: command
              }
            }]
          }
        },
        aggs: {
          filteredProc: {
            terms: {
              field: _constants2.PROCESS_COMMANDLINE_FIELD,
              size: 1
            },
            aggs: {
              timeseries: {
                date_histogram: {
                  field: _constants2.TIMESTAMP_FIELD,
                  fixed_interval: '1m',
                  extended_bounds: {
                    min: from,
                    max: to
                  }
                },
                aggs: {
                  cpu_by_state: {
                    terms: {
                      field: 'attributes.state',
                      size: 20
                    },
                    aggs: {
                      avg_cpu: {
                        avg: {
                          field: 'process.cpu.utilization'
                        }
                      }
                    }
                  },
                  cpu_total: {
                    sum_bucket: {
                      buckets_path: 'cpu_by_state>avg_cpu'
                    }
                  },
                  memory: {
                    avg: {
                      field: 'process.memory.utilization'
                    }
                  }
                }
              }
            }
          }
        }
      }
    }
  });
  const {
    buckets
  } = response.aggregations.process.filteredProc;
  const initialResponse = {
    cpu: {
      id: 'cpu',
      columns: TS_COLUMNS,
      rows: []
    },
    memory: {
      id: 'memory',
      columns: TS_COLUMNS,
      rows: []
    }
  };
  const timeseries = buckets.length > 0 ? buckets[0].timeseries.buckets.reduce((tsResult, tsBucket) => {
    tsResult.cpu.rows.push({
      metric_0: tsBucket.cpu_total.value,
      timestamp: tsBucket.key
    });
    tsResult.memory.rows.push({
      metric_0: tsBucket.memory.value !== null ? tsBucket.memory.value / 100 : null,
      // convert to ratio (0-1)
      timestamp: tsBucket.key
    });
    return tsResult;
  }, initialResponse) : initialResponse;
  return timeseries;
};
const getProcessListChart = async (infraMetricsClient, {
  hostTerm,
  indexPattern,
  to,
  command,
  schema
}) => {
  const detectedSchema = schema || _common.DataSchemaFormatEnum.ECS;
  if (detectedSchema === _common.DataSchemaFormatEnum.SEMCONV) {
    return await getSemConvProcessListChart(infraMetricsClient, {
      hostTerm,
      indexPattern,
      to,
      command
    });
  } else {
    return await getEcsProcessListChart(infraMetricsClient, {
      hostTerm,
      indexPattern,
      to,
      command
    });
  }
};
exports.getProcessListChart = getProcessListChart;