"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MetadataReceiver = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _utils = require("../utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class MetadataReceiver {
  constructor(logger, esClient) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    this.esClient = esClient;
    this.logger = logger.get(MetadataReceiver.name);
  }
  async getIndices() {
    this.logger.debug('Fetching indices');
    const request = {
      index: '*',
      expand_wildcards: ['open', 'hidden'],
      filter_path: ['*.mappings._source.mode', '*.settings.index.default_pipeline', '*.settings.index.final_pipeline', '*.settings.index.mode', '*.settings.index.provided_name']
    };
    return this.esClient.indices.get(request).then(indices => Object.entries(indices !== null && indices !== void 0 ? indices : {}).map(([index, value]) => {
      var _value$settings, _value$settings$index, _value$settings2, _value$settings2$inde, _value$settings3, _value$settings3$inde, _value$mappings, _value$mappings$_sour;
      return {
        index_name: index,
        default_pipeline: (_value$settings = value.settings) === null || _value$settings === void 0 ? void 0 : (_value$settings$index = _value$settings.index) === null || _value$settings$index === void 0 ? void 0 : _value$settings$index.default_pipeline,
        final_pipeline: (_value$settings2 = value.settings) === null || _value$settings2 === void 0 ? void 0 : (_value$settings2$inde = _value$settings2.index) === null || _value$settings2$inde === void 0 ? void 0 : _value$settings2$inde.final_pipeline,
        index_mode: (_value$settings3 = value.settings) === null || _value$settings3 === void 0 ? void 0 : (_value$settings3$inde = _value$settings3.index) === null || _value$settings3$inde === void 0 ? void 0 : _value$settings3$inde.mode,
        source_mode: (_value$mappings = value.mappings) === null || _value$mappings === void 0 ? void 0 : (_value$mappings$_sour = _value$mappings._source) === null || _value$mappings$_sour === void 0 ? void 0 : _value$mappings$_sour.mode
      };
    })).catch(error => {
      this.logger.warn('Error fetching indices', {
        error
      });
      throw error;
    });
  }
  async getDataStreams() {
    this.logger.debug('Fetching datstreams');
    const request = {
      name: '*',
      expand_wildcards: ['open', 'hidden'],
      filter_path: ['data_streams.name', 'data_streams.indices']
    };
    return this.esClient.indices.getDataStream(request).then(response => {
      var _response$data_stream;
      const streams = (_response$data_stream = response.data_streams) !== null && _response$data_stream !== void 0 ? _response$data_stream : [];
      return streams.map(ds => {
        var _ds$indices$map, _ds$indices;
        return {
          datastream_name: ds.name,
          indices: (_ds$indices$map = (_ds$indices = ds.indices) === null || _ds$indices === void 0 ? void 0 : _ds$indices.map(index => {
            return {
              index_name: index.index_name,
              ilm_policy: index.ilm_policy
            };
          })) !== null && _ds$indices$map !== void 0 ? _ds$indices$map : []
        };
      });
    }).catch(error => {
      this.logger.error('Error fetching datastreams', {
        error
      });
      throw error;
    });
  }
  async *getIndicesStats(indices, chunkSize) {
    const safeChunkSize = Math.min(chunkSize, 3000);
    this.logger.debug('Fetching indices stats');
    const groupedIndices = this.chunkStringsByMaxLength(indices, safeChunkSize);
    this.logger.debug('Splitted indices into groups', {
      groups: groupedIndices.length,
      indices: indices.length
    });
    for (const group of groupedIndices) {
      const request = {
        index: group,
        level: 'indices',
        metric: ['docs', 'search', 'store'],
        expand_wildcards: ['open', 'hidden'],
        filter_path: ['indices.*.total.search.query_total', 'indices.*.total.search.query_time_in_millis', 'indices.*.total.docs.count', 'indices.*.total.docs.deleted', 'indices.*.total.store.size_in_bytes', 'indices.*.primaries.docs.count', 'indices.*.primaries.docs.deleted', 'indices.*.primaries.store.size_in_bytes', 'indices.*.total.indexing.index_failed', 'indices.*.total.indexing.index_failed_due_to_version_conflict']
      };
      try {
        const response = await this.esClient.indices.stats(request);
        for (const [indexName, stats] of Object.entries((_response$indices = response.indices) !== null && _response$indices !== void 0 ? _response$indices : {})) {
          var _response$indices, _stats$total, _stats$total$search, _stats$total2, _stats$total2$search, _stats$total3, _stats$total3$docs, _stats$total4, _stats$total4$docs, _stats$total5, _stats$total5$store, _stats$total6, _stats$total6$indexin, _stats$total7, _stats$total7$indexin, _stats$primaries, _stats$primaries$docs, _stats$primaries2, _stats$primaries2$doc, _stats$primaries3, _stats$primaries3$sto;
          yield {
            index_name: indexName,
            query_total: (_stats$total = stats.total) === null || _stats$total === void 0 ? void 0 : (_stats$total$search = _stats$total.search) === null || _stats$total$search === void 0 ? void 0 : _stats$total$search.query_total,
            query_time_in_millis: (_stats$total2 = stats.total) === null || _stats$total2 === void 0 ? void 0 : (_stats$total2$search = _stats$total2.search) === null || _stats$total2$search === void 0 ? void 0 : _stats$total2$search.query_time_in_millis,
            docs_count: (_stats$total3 = stats.total) === null || _stats$total3 === void 0 ? void 0 : (_stats$total3$docs = _stats$total3.docs) === null || _stats$total3$docs === void 0 ? void 0 : _stats$total3$docs.count,
            docs_deleted: (_stats$total4 = stats.total) === null || _stats$total4 === void 0 ? void 0 : (_stats$total4$docs = _stats$total4.docs) === null || _stats$total4$docs === void 0 ? void 0 : _stats$total4$docs.deleted,
            docs_total_size_in_bytes: (_stats$total5 = stats.total) === null || _stats$total5 === void 0 ? void 0 : (_stats$total5$store = _stats$total5.store) === null || _stats$total5$store === void 0 ? void 0 : _stats$total5$store.size_in_bytes,
            index_failed: (_stats$total6 = stats.total) === null || _stats$total6 === void 0 ? void 0 : (_stats$total6$indexin = _stats$total6.indexing) === null || _stats$total6$indexin === void 0 ? void 0 : _stats$total6$indexin.index_failed,
            index_failed_due_to_version_conflict: (_stats$total7 = stats.total) === null || _stats$total7 === void 0 ? void 0 : (_stats$total7$indexin = _stats$total7.indexing) === null || _stats$total7$indexin === void 0 ? void 0 : _stats$total7$indexin.index_failed_due_to_version_conflict,
            docs_count_primaries: (_stats$primaries = stats.primaries) === null || _stats$primaries === void 0 ? void 0 : (_stats$primaries$docs = _stats$primaries.docs) === null || _stats$primaries$docs === void 0 ? void 0 : _stats$primaries$docs.count,
            docs_deleted_primaries: (_stats$primaries2 = stats.primaries) === null || _stats$primaries2 === void 0 ? void 0 : (_stats$primaries2$doc = _stats$primaries2.docs) === null || _stats$primaries2$doc === void 0 ? void 0 : _stats$primaries2$doc.deleted,
            docs_total_size_in_bytes_primaries: (_stats$primaries3 = stats.primaries) === null || _stats$primaries3 === void 0 ? void 0 : (_stats$primaries3$sto = _stats$primaries3.store) === null || _stats$primaries3$sto === void 0 ? void 0 : _stats$primaries3$sto.size_in_bytes
          };
        }
      } catch (error) {
        this.logger.error('Error fetching indices stats', {
          error
        });
        throw error;
      }
    }
  }
  async isIlmStatsAvailable() {
    const request = {
      index: '-invalid-index',
      only_managed: false,
      filter_path: ['indices.*.phase', 'indices.*.age', 'indices.*.policy']
    };
    const result = await this.esClient.ilm.explainLifecycle(request).then(() => {
      return true;
    }).catch(error => {
      return error.meta.statusCode === 404;
    });
    return result;
  }
  async *getIlmsStats(indices) {
    const groupedIndices = this.chunkStringsByMaxLength(indices);
    this.logger.debug('Splitted ilms into groups', {
      groups: groupedIndices.length,
      indices: indices.length
    });
    for (const group of groupedIndices) {
      const request = {
        index: group.join(','),
        only_managed: false,
        filter_path: ['indices.*.phase', 'indices.*.age', 'indices.*.policy']
      };
      const data = await this.esClient.ilm.explainLifecycle(request);
      try {
        for (const [indexName, stats] of Object.entries((_data$indices = data.indices) !== null && _data$indices !== void 0 ? _data$indices : {})) {
          var _data$indices;
          const entry = {
            index_name: indexName,
            phase: 'phase' in stats && stats.phase || undefined,
            age: 'age' in stats && stats.age || undefined,
            policy_name: 'policy' in stats && stats.policy || undefined
          };
          yield entry;
        }
      } catch (error) {
        this.logger.error('Error fetching ilm stats', {
          error
        });
        throw error;
      }
    }
  }
  async getIndexTemplatesStats() {
    this.logger.debug('Fetching index templates');
    const request = {
      name: '*',
      filter_path: ['index_templates.name', 'index_templates.index_template.template.settings.index.mode', 'index_templates.index_template.data_stream', 'index_templates.index_template._meta.package.name', 'index_templates.index_template._meta.managed_by', 'index_templates.index_template._meta.beat', 'index_templates.index_template._meta.managed', 'index_templates.index_template.composed_of', 'index_templates.index_template.template.mappings._source.enabled', 'index_templates.index_template.template.mappings._source.includes', 'index_templates.index_template.template.mappings._source.excludes']
    };
    return this.esClient.indices.getIndexTemplate(request).then(response => {
      var _response$index_templ;
      const templates = (_response$index_templ = response.index_templates) !== null && _response$index_templ !== void 0 ? _response$index_templ : [];
      return templates.map(props => {
        var _props$index_template, _props$index_template2, _props$index_template3, _props$index_template4, _props$index_template5, _props$index_template6, _props$index_template7, _props$index_template8, _props$index_template9, _props$index_template10, _props$index_template11, _props$index_template12, _props$index_template13, _props$index_template14, _props$index_template15, _props$index_template16, _props$index_template17, _props$index_template18, _props$index_template19, _props$index_template20;
        const datastream = ((_props$index_template = props.index_template) === null || _props$index_template === void 0 ? void 0 : _props$index_template.data_stream) !== undefined;
        return {
          template_name: props.name,
          index_mode: (_props$index_template2 = props.index_template.template) === null || _props$index_template2 === void 0 ? void 0 : (_props$index_template3 = _props$index_template2.settings) === null || _props$index_template3 === void 0 ? void 0 : (_props$index_template4 = _props$index_template3.index) === null || _props$index_template4 === void 0 ? void 0 : _props$index_template4.mode,
          package_name: (_props$index_template5 = props.index_template._meta) === null || _props$index_template5 === void 0 ? void 0 : (_props$index_template6 = _props$index_template5.package) === null || _props$index_template6 === void 0 ? void 0 : _props$index_template6.name,
          datastream,
          managed_by: (_props$index_template7 = props.index_template._meta) === null || _props$index_template7 === void 0 ? void 0 : _props$index_template7.managed_by,
          beat: (_props$index_template8 = props.index_template._meta) === null || _props$index_template8 === void 0 ? void 0 : _props$index_template8.beat,
          is_managed: (_props$index_template9 = props.index_template._meta) === null || _props$index_template9 === void 0 ? void 0 : _props$index_template9.managed,
          composed_of: props.index_template.composed_of,
          source_enabled: (_props$index_template10 = props.index_template.template) === null || _props$index_template10 === void 0 ? void 0 : (_props$index_template11 = _props$index_template10.mappings) === null || _props$index_template11 === void 0 ? void 0 : (_props$index_template12 = _props$index_template11._source) === null || _props$index_template12 === void 0 ? void 0 : _props$index_template12.enabled,
          source_includes: (_props$index_template13 = (_props$index_template14 = props.index_template.template) === null || _props$index_template14 === void 0 ? void 0 : (_props$index_template15 = _props$index_template14.mappings) === null || _props$index_template15 === void 0 ? void 0 : (_props$index_template16 = _props$index_template15._source) === null || _props$index_template16 === void 0 ? void 0 : _props$index_template16.includes) !== null && _props$index_template13 !== void 0 ? _props$index_template13 : [],
          source_excludes: (_props$index_template17 = (_props$index_template18 = props.index_template.template) === null || _props$index_template18 === void 0 ? void 0 : (_props$index_template19 = _props$index_template18.mappings) === null || _props$index_template19 === void 0 ? void 0 : (_props$index_template20 = _props$index_template19._source) === null || _props$index_template20 === void 0 ? void 0 : _props$index_template20.excludes) !== null && _props$index_template17 !== void 0 ? _props$index_template17 : []
        };
      });
    }).catch(error => {
      this.logger.warn('Error fetching index templates', {
        error
      });
      throw error;
    });
  }
  async *getIlmsPolicies(ilms, chunkSize) {
    const safeChunkSize = Math.min(chunkSize, 3000);
    const phase = obj => {
      let value;
      if (obj !== null && obj !== undefined && typeof obj === 'object' && 'min_age' in obj) {
        value = {
          min_age: obj.min_age
        };
      }
      return value;
    };
    const groupedIlms = this.chunkStringsByMaxLength(ilms, safeChunkSize);
    this.logger.debug('Splitted ilms into groups', {
      groups: groupedIlms.length,
      ilms: ilms.length
    });
    for (const group of groupedIlms) {
      this.logger.debug('Fetching ilm policies');
      const request = {
        name: group.join(','),
        filter_path: ['*.policy.phases.cold.min_age', '*.policy.phases.delete.min_age', '*.policy.phases.frozen.min_age', '*.policy.phases.hot.min_age', '*.policy.phases.warm.min_age', '*.modified_date']
      };
      const response = await this.esClient.ilm.getLifecycle(request);
      try {
        for (const [policyName, stats] of Object.entries(response !== null && response !== void 0 ? response : {})) {
          yield {
            policy_name: policyName,
            modified_date: stats.modified_date,
            phases: {
              cold: phase(stats.policy.phases.cold),
              delete: phase(stats.policy.phases.delete),
              frozen: phase(stats.policy.phases.frozen),
              hot: phase(stats.policy.phases.hot),
              warm: phase(stats.policy.phases.warm)
            }
          };
        }
      } catch (error) {
        this.logger.error('Error fetching ilm policies', {
          error
        });
        throw error;
      }
    }
  }
  chunkStringsByMaxLength(strings, maxLength = 3072) {
    // plus 1 for the comma separator
    return (0, _utils.chunkedBy)(strings, maxLength, index => index.length + 1);
  }
}
exports.MetadataReceiver = MetadataReceiver;